Gnome::Gdk3::Window
===================

Onscreen display areas in the target window system

Description
===========

A **Gnome::Gdk3::Window** is a (usually) rectangular region on the screen. It’s a low-level object, used to implement high-level objects such as **Gnome::Gtk3::Widget** and **Gnome::Gtk3::Window** on the GTK+ level. A **Gnome::Gtk3::Window** is a toplevel window, the thing a user might think of as a “window” with a titlebar and so on; a **Gnome::Gtk3::Window** may contain many **Gnome::Gdk3::Windows**. For example, each **Gnome::Gtk3::Button** has a **Gnome::Gdk3::Window** associated with it.

Composited Windows
------------------

Normally, the windowing system takes care of rendering the contents of a child window onto its parent window. This mechanism can be intercepted by calling `gdk_window_set_composited()` on the child window. For a “composited” window it is the responsibility of the application to render the window contents at the right spot.

Offscreen Windows
-----------------

Offscreen windows are more general than composited windows, since they allow not only to modify the rendering of the child window onto its parent, but also to apply coordinate transformations.

To integrate an offscreen window into a window hierarchy, one has to call `gdk_offscreen_window_set_embedder()` and handle a number of signals. The *pick-embedded-child* signal on the embedder window is used to select an offscreen child at given coordinates, and the *to-embedder* and *from-embedder* signals on the offscreen window are used to translate coordinates between the embedder and the offscreen window.

For rendering an offscreen window onto its embedder, the contents of the offscreen window are available as a surface, via `gdk_offscreen_window_get_surface()`.

Testing
-------

Some routines are added to this module for GUI testing. They were in a separate C file but could be included here. These are test utilities to test some of the events generated by mouse and keyboard. This set of routines will not be of much use for the GUI developer. However, for interface testing, this module is may be of use. A package which will be using this, is in development and is called **Gnome::T**.

Synopsis
========

Declaration
-----------

    unit class Gnome::Gdk3::Window;
    also is Gnome::GObject::Object;

Types
=====

enum GdkAnchorHints
-------------------

Positioning hints for aligning a window relative to a rectangle.

These hints determine how the window should be positioned in the case that the window would fall off-screen if placed in its ideal position.

For example, `GDK_ANCHOR_FLIP_X` will replace `GDK_GRAVITY_NORTH_WEST` with `GDK_GRAVITY_NORTH_EAST` and vice versa if the window extends beyond the left or right edges of the monitor.

If `GDK_ANCHOR_SLIDE_X` is set, the window can be shifted horizontally to fit on-screen. If `GDK_ANCHOR_RESIZE_X` is set, the window can be shrunken horizontally to fit.

In general, when multiple flags are set, flipping should take precedence over sliding, which should take precedence over resizing.

Stability: Unstable

  * GDK_ANCHOR_FLIP_X: allow flipping anchors horizontally

  * GDK_ANCHOR_FLIP_Y: allow flipping anchors vertically

  * GDK_ANCHOR_SLIDE_X: allow sliding window horizontally

  * GDK_ANCHOR_SLIDE_Y: allow sliding window vertically

  * GDK_ANCHOR_RESIZE_X: allow resizing window horizontally

  * GDK_ANCHOR_RESIZE_Y: allow resizing window vertically

  * GDK_ANCHOR_FLIP: allow flipping anchors on both axes

  * GDK_ANCHOR_SLIDE: allow sliding window on both axes

  * GDK_ANCHOR_RESIZE: allow resizing window on both axes

enum GdkFullscreenMode
----------------------

Indicates which monitor (in a multi-head setup) a window should span over when in fullscreen mode.

  * GDK_FULLSCREEN_ON_CURRENT_MONITOR: Fullscreen on current monitor only.

  * GDK_FULLSCREEN_ON_ALL_MONITORS: Span across all monitors when fullscreen.

enum GdkGravity
---------------

Defines the reference point of a window and the meaning of coordinates passed to `gtk_window_move()`. See `gtk_window_move()` and the "implementation notes" section of the [Extended Window Manager Hints](http://www.freedesktop.org/Standards/wm-spec) specification for more details.

  * GDK_GRAVITY_NORTH_WEST: the reference point is at the top left corner.

  * GDK_GRAVITY_NORTH: the reference point is in the middle of the top edge.

  * GDK_GRAVITY_NORTH_EAST: the reference point is at the top right corner.

  * GDK_GRAVITY_WEST: the reference point is at the middle of the left edge.

  * GDK_GRAVITY_CENTER: the reference point is at the center of the window.

  * GDK_GRAVITY_EAST: the reference point is at the middle of the right edge.

  * GDK_GRAVITY_SOUTH_WEST: the reference point is at the lower left corner.

  * GDK_GRAVITY_SOUTH: the reference point is at the middle of the lower edge.

  * GDK_GRAVITY_SOUTH_EAST: the reference point is at the lower right corner.

  * GDK_GRAVITY_STATIC: the reference point is at the top left corner of the window itself, ignoring window manager decorations.

enum GdkWMDecoration
--------------------

These are hints originally defined by the Motif toolkit. The window manager can use them when determining how to decorate the window. The hint must be set before mapping the window.

  * GDK_DECOR_ALL: all decorations should be applied.

  * GDK_DECOR_BORDER: a frame should be drawn around the window.

  * GDK_DECOR_RESIZEH: the frame should have resize handles.

  * GDK_DECOR_TITLE: a titlebar should be placed above the window.

  * GDK_DECOR_MENU: a button for opening a menu should be included.

  * GDK_DECOR_MINIMIZE: a minimize button should be included.

  * GDK_DECOR_MAXIMIZE: a maximize button should be included.

enum GdkWMFunction
------------------

These are hints originally defined by the Motif toolkit. The window manager can use them when determining the functions to offer for the window. The hint must be set before mapping the window.

  * GDK_FUNC_ALL: all functions should be offered.

  * GDK_FUNC_RESIZE: the window should be resizable.

  * GDK_FUNC_MOVE: the window should be movable.

  * GDK_FUNC_MINIMIZE: the window should be minimizable.

  * GDK_FUNC_MAXIMIZE: the window should be maximizable.

  * GDK_FUNC_CLOSE: the window should be closable.

enum GdkWindowAttributesType
----------------------------

Used to indicate which fields in the **Gnome::Gdk3::WindowAttr** struct should be honored. For example, if you filled in the “cursor” and “x” fields of **Gnome::Gdk3::WindowAttr**, pass “*GDK_WA_X* | *GDK_WA_CURSOR*” to `new()`. Fields in **Gnome::Gdk3::WindowAttr** not covered by a bit in this enum are required; for example, the *width*/*height*, *wclass*, and *window_type* fields are required, they have no corresponding flag in **Gnome::Gdk3::WindowAttributesType**.

  * GDK_WA_TITLE: Honor the title field

  * GDK_WA_X: Honor the X coordinate field

  * GDK_WA_Y: Honor the Y coordinate field

  * GDK_WA_CURSOR: Honor the cursor field

  * GDK_WA_VISUAL: Honor the visual field

  * GDK_WA_WMCLASS: Honor the wmclass_class and wmclass_name fields

  * GDK_WA_NOREDIR: Honor the override_redirect field

  * GDK_WA_TYPE_HINT: Honor the type_hint field

enum GdkWindowEdge
------------------

Determines a window edge or corner.

  * GDK_WINDOW_EDGE_NORTH_WEST: the top left corner.

  * GDK_WINDOW_EDGE_NORTH: the top edge.

  * GDK_WINDOW_EDGE_NORTH_EAST: the top right corner.

  * GDK_WINDOW_EDGE_WEST: the left edge.

  * GDK_WINDOW_EDGE_EAST: the right edge.

  * GDK_WINDOW_EDGE_SOUTH_WEST: the lower left corner.

  * GDK_WINDOW_EDGE_SOUTH: the lower edge.

  * GDK_WINDOW_EDGE_SOUTH_EAST: the lower right corner.

enum GdkWindowHints
-------------------

Used to indicate which fields of a **Gnome::Gdk3::Geometry** struct should be paid attention to. Also, the presence/absence of *GDK_HINT_POS*, *GDK_HINT_USER_POS*, and *GDK_HINT_USER_SIZE* is significant, though they don't directly refer to **Gnome::Gdk3::Geometry** fields. *GDK_HINT_USER_POS* will be set automatically by **Gnome::Gdk3::Window** if you call `gtk_window_move()`. *GDK_HINT_USER_POS* and *GDK_HINT_USER_SIZE* should be set if the user specified a size/position using a --geometry command-line argument; `gtk_window_parse_geometry()` automatically sets these flags.

  * GDK_HINT_POS: indicates that the program has positioned the window

  * GDK_HINT_MIN_SIZE: min size fields are set

  * GDK_HINT_MAX_SIZE: max size fields are set

  * GDK_HINT_BASE_SIZE: base size fields are set

  * GDK_HINT_ASPECT: aspect ratio fields are set

  * GDK_HINT_RESIZE_INC: resize increment fields are set

  * GDK_HINT_WIN_GRAVITY: window gravity field is set

  * GDK_HINT_USER_POS: indicates that the window’s position was explicitly set by the user

  * GDK_HINT_USER_SIZE: indicates that the window’s size was explicitly set by the user

enum GdkWindowType
------------------

Describes the kind of window.

  * GDK_WINDOW_ROOT: root window; this window has no parent, covers the entire screen, and is created by the window system

  * GDK_WINDOW_TOPLEVEL: toplevel window (used to implement **Gnome::Gdk3::Window**)

  * GDK_WINDOW_CHILD: child window (used to implement e.g. **Gnome::Gdk3::Entry**)

  * GDK_WINDOW_TEMP: override redirect temporary window (used to implement **Gnome::Gdk3::Menu**)

  * GDK_WINDOW_FOREIGN: foreign window (see `foreign_new()`)

  * GDK_WINDOW_OFFSCREEN: offscreen window (see [Offscreen Windows][OFFSCREEN-WINDOWS]). Since 2.18

  * GDK_WINDOW_SUBSURFACE: subsurface-based window; This window is visually tied to a toplevel, and is moved/stacked with it. Currently this window type is only implemented in Wayland. Since 3.14

enum GdkWindowWindowClass
-------------------------

*GDK_INPUT_OUTPUT* windows are the standard kind of window you might expect. Such windows receive events and are also displayed on screen. *GDK_INPUT_ONLY* windows are invisible; they are usually placed above other windows in order to trap or filter the events. You can’t draw on *GDK_INPUT_ONLY* windows.

  * GDK_INPUT_OUTPUT: window for graphics and events

  * GDK_INPUT_ONLY: window for events only

class N-GdkGeometry
-------------------

The **Gnome::Gdk3::Geometry** struct gives the window manager information about a window’s geometry constraints. Normally you would set these on the GTK+ level using `gtk_window_set_geometry_hints()`. **Gnome::Gdk3::Window** then sets the hints on the **Gnome::Gdk3::Window** it creates.

`set_geometry_hints()` expects the hints to be fully valid already and simply passes them to the window manager; in contrast, `gtk_window_set_geometry_hints()` performs some interpretation. For example, **Gnome::Gdk3::Window** will apply the hints to the geometry widget instead of the toplevel window, if you set a geometry widget. Also, the *min_width*/*min_height*/*max_width*/*max_height* fields may be set to -1, and **Gnome::Gdk3::Window** will substitute the size request of the window or geometry widget. If the minimum size hint is not provided, **Gnome::Gdk3::Window** will use its requisition as the minimum size. If the minimum size is provided and a geometry widget is set, **Gnome::Gdk3::Window** will take the minimum size as the minimum size of the geometry widget rather than the entire window. The base size is treated similarly.

The canonical use-case for `gtk_window_set_geometry_hints()` is to get a terminal widget to resize properly. Here, the terminal text area should be the geometry widget; **Gnome::Gdk3::Window** will then automatically set the base size to the size of other widgets in the terminal window, such as the menubar and scrollbar. Then, the *width_inc* and *height_inc* fields should be set to the size of one character in the terminal. Finally, the base size should be set to the size of one character. The net effect is that the minimum size of the terminal will have a 1x1 character terminal area, and only terminal sizes on the “character grid” will be allowed.

Here’s an example of how the terminal example would be implemented, assuming a terminal area widget called “terminal” and a toplevel window “toplevel”:

The other useful fields are the *min_aspect* and *max_aspect* fields; these contain a width/height ratio as a floating point number. If a geometry widget is set, the aspect applies to the geometry widget rather than the entire window. The most common use of these hints is probably to set *min_aspect* and *max_aspect* to the same value, thus forcing the window to keep a constant aspect ratio.

  * `Int()` $.min_width: minimum width of window (or -1 to use requisition, with **Gnome::Gdk3::Window** only)

  * `Int()` $.min_height: minimum height of window (or -1 to use requisition, with **Gnome::Gdk3::Window** only)

  * `Int()` $.max_width: maximum width of window (or -1 to use requisition, with **Gnome::Gdk3::Window** only)

  * `Int()` $.max_height: maximum height of window (or -1 to use requisition, with **Gnome::Gdk3::Window** only)

  * `Int()` $.base_width: allowed window widths are *base_width* + *width_inc* * N where N is any integer (-1 allowed with **Gnome::Gdk3::Window**)

  * `Int()` $.base_height: allowed window widths are *base_height* + *height_inc* * N where N is any integer (-1 allowed with **Gnome::Gdk3::Window**)

  * `Int()` $.width_inc: width resize increment

  * `Int()` $.height_inc: height resize increment

  * `Num()` $.min_aspect: minimum width/height ratio

  * `Num()` $.max_aspect: maximum width/height ratio

  * GdkGravity $.win_gravity: window gravity, see `gtk_window_set_gravity()`

class N-GdkWindowAttr
---------------------

Attributes to use for a newly-created window.

  * Str $.title: title of the window (for toplevel windows)

  * `Int()` $.event_mask: event mask (see `set_events()`)

  * `Int()` $.x: X coordinate relative to parent window (see `move()`)

  * `Int()` $.y: Y coordinate relative to parent window (see `move()`)

  * `Int()` $.width: width of window

  * `Int()` $.height: height of window

  * GdkWindowWindowClass $.wclass: **Gnome::Gdk3::DK_INPUT_OUTPUT** (normal window) or **Gnome::Gdk3::DK_INPUT_ONLY** (invisible window that receives events)

  * N-GObject $.visual: **Gnome::Gdk3::Visual** for window

  * GdkWindowType $.window_type: type of window

  * N-GObject $.cursor: cursor for the window (see `set_cursor()`)

  * Str $.wmclass_name: don’t use (see `gtk_window_set_wmclass()`)

  * Str $.wmclass_class: don’t use (see `gtk_window_set_wmclass()`)

  * Bool $.override_redirect: `True` to bypass the window manager

  * GdkWindowTypeHint $.type_hint: a hint of the function of the window

Methods
=======

new
---

beep
----

Emits a short beep associated to *window* in the appropriate display, if supported. Otherwise, emits a short beep on the display just as `Gnome::Gdk3::Display.beep()`.

    method beep ( )

begin-move-drag
---------------

Begins a window move operation (for a toplevel window).

This function assumes that the drag is controlled by the client pointer device, use `begin_move_drag_for_device()` to begin a drag with a different device.

    method begin-move-drag (
      Int() $button, Int() $root_x, Int() $root_y, UInt $timestamp
    )

  * $button; the button being used to drag, or 0 for a keyboard-initiated drag

  * $root_x; root window X coordinate of mouse click that began the drag

  * $root_y; root window Y coordinate of mouse click that began the drag

  * $timestamp; timestamp of mouse click that began the drag

begin-move-drag-for-device
--------------------------

Begins a window move operation (for a toplevel window). You might use this function to implement a “window move grip,” for example. The function works best with window managers that support the [Extended Window Manager Hints](http://www.freedesktop.org/Standards/wm-spec) but has a fallback implementation for other window managers.

    method begin-move-drag-for-device (
      N-GObject() $device, Int() $button,
      Int() $root_x, Int() $root_y, UInt $timestamp
    )

  * $device; the device used for the operation

  * $button; the button being used to drag, or 0 for a keyboard-initiated drag

  * $root_x; root window X coordinate of mouse click that began the drag

  * $root_y; root window Y coordinate of mouse click that began the drag

  * $timestamp; timestamp of mouse click that began the drag

begin-resize-drag
-----------------

Begins a window resize operation (for a toplevel window).

This function assumes that the drag is controlled by the client pointer device, use `begin_resize_drag_for_device()` to begin a drag with a different device.

    method begin-resize-drag (
      GdkWindowEdge $edge, Int() $button,
      Int() $root_x, Int() $root_y, UInt $timestamp
    )

  * $edge; the edge or corner from which the drag is started

  * $button; the button being used to drag, or 0 for a keyboard-initiated drag

  * $root_x; root window X coordinate of mouse click that began the drag

  * $root_y; root window Y coordinate of mouse click that began the drag

  * $timestamp; timestamp of mouse click that began the drag (use `gdk_event_get_time()`)

begin-resize-drag-for-device
----------------------------

Begins a window resize operation (for a toplevel window). You might use this function to implement a “window resize grip,” for example; in fact **Gnome::Gdk3::Statusbar** uses it. The function works best with window managers that support the [Extended Window Manager Hints](http://www.freedesktop.org/Standards/wm-spec) but has a fallback implementation for other window managers.

    method begin-resize-drag-for-device (
      GdkWindowEdge $edge, N-GObject() $device, Int() $button,
      Int() $root_x, Int() $root_y, UInt $timestamp
    )

  * $edge; the edge or corner from which the drag is started

  * $device; the device used for the operation

  * $button; the button being used to drag, or 0 for a keyboard-initiated drag

  * $root_x; root window X coordinate of mouse click that began the drag

  * $root_y; root window Y coordinate of mouse click that began the drag

  * $timestamp; timestamp of mouse click that began the drag (use `gdk_event_get_time()`)

constrain-size
--------------

Constrains a desired width and height according to a set of geometry hints (such as minimum and maximum size).

    method constrain-size (
      N-GdkGeometry $geometry, GdkWindowHints $flags,
      Int() $width, Int() $height
    )

  * $geometry; a **Gnome::Gdk3::Geometry** structure

  * $flags; a mask indicating what portions of *geometry* are set

  * $width; desired width of window

  * $height; desired height of the window

  * $new_width; location to store resulting width

  * $new_height; location to store resulting height

coords-from-parent
------------------

Transforms window coordinates from a parent window to a child window, where the parent window is the normal parent as returned by `get_parent()` for normal windows, and the window's embedder as returned by `gdk_offscreen_window_get_embedder()` for offscreen windows.

For normal windows, calling this function is equivalent to subtracting the return values of `get_position()` from the parent coordinates. For offscreen windows however (which can be arbitrarily transformed), this function calls the GdkWindow::from-embedder: signal to translate the coordinates.

You should always use this function when writing generic code that walks down a window hierarchy.

See also: `coords_to_parent()`

    method coords-from-parent (
      Num() $parent_x, Num() $parent_y, Num() $x, Num() $y
    )

  * $parent_x; X coordinate in parent’s coordinate system

  * $parent_y; Y coordinate in parent’s coordinate system

  * $x; return location for X coordinate in child’s coordinate system

  * $y; return location for Y coordinate in child’s coordinate system

coords-to-parent
----------------

Transforms window coordinates from a child window to its parent window, where the parent window is the normal parent as returned by `get_parent()` for normal windows, and the window's embedder as returned by `gdk_offscreen_window_get_embedder()` for offscreen windows.

For normal windows, calling this function is equivalent to adding the return values of `get_position()` to the child coordinates. For offscreen windows however (which can be arbitrarily transformed), this function calls the GdkWindow::to-embedder: signal to translate the coordinates.

You should always use this function when writing generic code that walks up a window hierarchy.

See also: `coords_from_parent()`

    method coords-to-parent (
      Num() $x, Num() $y, Num() $parent_x, Num() $parent_y
    )

  * $x; X coordinate in child’s coordinate system

  * $y; Y coordinate in child’s coordinate system

  * $parent_x; return location for X coordinate in parent’s coordinate system, or `undefined`

  * $parent_y; return location for Y coordinate in parent’s coordinate system, or `undefined`

create-similar-image-surface
----------------------------

Create a new image surface that is efficient to draw on the given *window*.

Initially the surface contents are all 0 (transparent if contents have transparency, black otherwise.)

The *width* and *height* of the new surface are not affected by the scaling factor of the *window*, or by the *scale* argument; they are the size of the surface in device pixels. If you wish to create an image surface capable of holding the contents of *window* you can use:

Note that unlike `cairo_surface_create_similar_image()`, the new surface's device scale is set to *scale*, or to the scale factor of *window* if *scale* is 0.

Returns: a pointer to the newly allocated surface. The caller owns the surface and should call `cairo_surface_destroy()` when done with it.

This function always returns a valid pointer, but it will return a pointer to a “nil” surface if *other* is already in an error state or any other error occurs.

    method create-similar-image-surface (
      cairo_format_t $format, Int() $width, Int() $height, Int() $scale
      --> cairo_surface_t
    )

  * $format; (type int): the format for the new surface

  * $width; width of the new surface

  * $height; height of the new surface

  * $scale; the scale of the new surface, or 0 to use same as *window*

create-similar-surface
----------------------

Create a new surface that is as compatible as possible with the given *window*. For example the new surface will have the same fallback resolution and font options as *window*. Generally, the new surface will also use the same backend as *window*, unless that is not possible for some reason. The type of the returned surface may be examined with `cairo_surface_get_type()`.

Initially the surface contents are all 0 (transparent if contents have transparency, black otherwise.)

Returns: a pointer to the newly allocated surface. The caller owns the surface and should call `cairo_surface_destroy()` when done with it.

This function always returns a valid pointer, but it will return a pointer to a “nil” surface if *other* is already in an error state or any other error occurs.

    method create-similar-surface (
      cairo_content_t $content, Int() $width, Int() $height
      --> cairo_surface_t
    )

  * $content; the content for the new surface

  * $width; width of the new surface

  * $height; height of the new surface

deiconify
---------

Attempt to deiconify (unminimize) *window*. On X11 the window manager may choose to ignore the request to deiconify. When using GTK+, use `gtk_window_deiconify()` instead of the **Gnome::Gdk3::Window** variant. Or better yet, you probably want to use `gtk_window_present_with_time()`, which raises the window, focuses it, unminimizes it, and puts it on the current desktop.

    method deiconify ( )

destroy
-------

Destroys the window system resources associated with *window* and decrements *window*'s reference count. The window system resources for all children of *window* are also destroyed, but the children’s reference counts are not decremented.

Note that a window will not be destroyed automatically when its reference count reaches zero. You must call this function yourself before that happens.

After this call the object becomes invalid and its native object lost.

    method destroy ( )

ensure-native
-------------

Tries to ensure that there is a window-system native window for this GdkWindow. This may fail in some situations, returning `False`.

Offscreen window and children of them can never have native windows.

Some backends may not support native child windows.

Returns: `True` if the window has a native window, `False` otherwise

    method ensure-native ( --> Bool )

focus
-----

Sets keyboard focus to *window*. In most cases, `Gnome::Gtk3::Window.present-with-time()` should be used on a **Gnome::Gtk3::Window**, rather than calling this function.

    method focus ( UInt $timestamp )

  * $timestamp; timestamp of the event triggering the window focus

freeze-updates
--------------

Temporarily freezes a window such that it won’t receive expose events. The window will begin receiving expose events again when `thaw_updates()` is called. If `freeze_updates()` has been called more than once, `thaw_updates()` must be called an equal number of times to begin processing exposes.

    method freeze-updates ( )

fullscreen
----------

Moves the window into fullscreen mode. This means the window covers the entire screen and is above any panels or task bars.

If the window was already fullscreen, then this function does nothing.

On X11, asks the window manager to put *window* in a fullscreen state, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “fullscreen”; so you can’t rely on the fullscreenification actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

    method fullscreen ( )

fullscreen-on-monitor
---------------------

Moves the window into fullscreen mode on the given monitor. This means the window covers the entire screen and is above any panels or task bars.

If the window was already fullscreen, then this function does nothing.

    method fullscreen-on-monitor ( Int() $monitor )

  * $monitor; Which monitor to display fullscreen on.

geometry-changed
----------------

This function informs GDK that the geometry of an embedded offscreen window has changed. This is necessary for GDK to keep track of which offscreen window the pointer is in.

    method geometry-changed ( )

get-accept-focus
----------------

Determines whether or not the desktop environment shuld be hinted that the window does not want to receive input focus.

Returns: whether or not the window should receive input focus.

    method get-accept-focus ( --> Bool )

get-children
------------

Gets the list of children of *window* known to GDK. This function only returns children created via GDK, so for example it’s useless when used with the root window; it only returns windows an application created itself.

The returned list must be freed, but the elements in the list need not be.

Returns: (transfer container) (element-type GdkWindow): list of child windows inside *window*

    method get-children ( --> N-GList )

get-children-with-user-data
---------------------------

Gets the list of children of *window* known to GDK with a particular *user_data* set on it.

The returned list must be freed, but the elements in the list need not be.

The list is returned in (relative) stacking order, i.e. the lowest window is first.

Returns: (transfer container) (element-type GdkWindow): list of child windows inside *window*

    method get-children-with-user-data ( Pointer $user_data --> N-GList )

  * $user_data; user data to look for

get-cursor
----------

Retrieves a **Gnome::Gdk3::Cursor** pointer for the cursor currently set on the specified **Gnome::Gdk3::Window**, or `undefined`. If the return value is `undefined` then there is no custom cursor set on the specified window, and it is using the cursor for its parent window.

Returns: a **Gnome::Gdk3::Cursor**, or `undefined`. The returned object is owned by the **Gnome::Gdk3::Window** and should not be unreferenced directly. Use `set_cursor()` to unset the cursor of the window

    method get-cursor ( --> N-GObject )

get-decorations
---------------

Returns the decorations set on the GdkWindow with `set_decorations()`.

Returns: `True` if the window has decorations set, `False` otherwise.

    method get-decorations ( GdkWMDecoration $decorations --> Bool )

  * $decorations; The window decorations will be written here

get-default-root-window
-----------------------

Obtains the root window (parent all other windows are inside) for the default display and screen.

Returns: the default root window

    method get-default-root-window ( --> N-GObject )

get-device-cursor
-----------------

Retrieves a **Gnome::Gdk3::Cursor** pointer for the *device* currently set on the specified **Gnome::Gdk3::Window**, or `undefined`. If the return value is `undefined` then there is no custom cursor set on the specified window, and it is using the cursor for its parent window.

Returns: a **Gnome::Gdk3::Cursor**, or `undefined`. The returned object is owned by the **Gnome::Gdk3::Window** and should not be unreferenced directly. Use `set_cursor()` to unset the cursor of the window

    method get-device-cursor ( N-GObject() $device --> N-GObject )

  * $device; a master, pointer **Gnome::Gdk3::Device**.

get-device-events
-----------------

Returns the event mask for *window* corresponding to an specific device.

Returns: device event mask for *window*

    method get-device-events ( N-GObject() $device --> GdkEventMask )

  * $device; a **Gnome::Gdk3::Device**.

get-device-position
-------------------

Obtains the current device position and modifier state. The position is given in coordinates relative to the upper left corner of *window*.

Use `get_device_position_double()` if you need subpixel precision.

Returns: The window underneath *device* (as with `gdk_device_get_window_at_position()`), or `undefined` if the window is not known to GDK.

    method get-device-position (
      N-GObject() $device, GdkModifierType $mask --> N-GObject
    )

  * $device; pointer **Gnome::Gdk3::Device** to query to.

  * $x; return location for the X coordinate of *device*, or `undefined`.

  * $y; return location for the Y coordinate of *device*, or `undefined`.

  * $mask; return location for the modifier mask, or `undefined`.

get-device-position-double
--------------------------

Obtains the current device position in doubles and modifier state. The position is given in coordinates relative to the upper left corner of *window*.

Returns: The window underneath *device* (as with `gdk_device_get_window_at_position()`), or `undefined` if the window is not known to GDK.

    method get-device-position-double (
      N-GObject() $device, Num() $x, Num() $y, GdkModifierType $mask
      --> N-GObject
    )

  * $device; pointer **Gnome::Gdk3::Device** to query to.

  * $x; return location for the X coordinate of *device*, or `undefined`.

  * $y; return location for the Y coordinate of *device*, or `undefined`.

  * $mask; return location for the modifier mask, or `undefined`.

get-display
-----------

Gets the **Gnome::Gdk3::Display** associated with a **Gnome::Gdk3::Window**.

Returns: the **Gnome::Gdk3::Display** associated with *window*

    method get-display ( --> N-GObject )

get-effective-parent
--------------------

Obtains the parent of *window*, as known to GDK. Works like `get_parent()` for normal windows, but returns the window’s embedder for offscreen windows.

See also: `gdk_offscreen_window_get_embedder()`

Returns: effective parent of *window*

    method get-effective-parent ( --> N-GObject )

get-effective-toplevel
----------------------

Gets the toplevel window that’s an ancestor of *window*.

Works like `get_toplevel()`, but treats an offscreen window's embedder as its parent, using `get_effective_parent()`.

See also: `gdk_offscreen_window_get_embedder()`

Returns: the effective toplevel window containing *window*

    method get-effective-toplevel ( --> N-GObject )

get-event-compression
---------------------

Get the current event compression setting for this window.

Returns: `True` if motion events will be compressed

    method get-event-compression ( --> Bool )

get-events
----------

Gets the event mask for *window* for all master input devices. See `set_events()`.

Returns: event mask for *window*

    method get-events ( --> GdkEventMask )

get-focus-on-map
----------------

Determines whether or not the desktop environment should be hinted that the window does not want to receive input focus when it is mapped.

Returns: whether or not the window wants to receive input focus when it is mapped.

    method get-focus-on-map ( --> Bool )

get-frame-clock
---------------

Gets the frame clock for the window. The frame clock for a window never changes unless the window is reparented to a new toplevel window.

Returns: the frame clock

    method get-frame-clock ( --> N-GObject )

get-frame-extents
-----------------

Obtains the bounding box of the window, including window manager titlebar/borders if any. The frame position is given in root window coordinates. To get the position of the window itself (rather than the frame) in root window coordinates, use `get_origin()`.

    method get-frame-extents ( N-GObject() $rect )

  * $rect; rectangle to fill with bounding box of the window frame

get-fullscreen-mode
-------------------

Obtains the **Gnome::Gdk3::FullscreenMode** of the *window*.

Returns: The **Gnome::Gdk3::FullscreenMode** applied to the window when fullscreen.

    method get-fullscreen-mode ( --> GdkFullscreenMode )

get-geometry
------------

Any of the return location arguments to this function may be `undefined`, if you aren’t interested in getting the value of that field.

The X and Y coordinates returned are relative to the parent window of *window*, which for toplevels usually means relative to the window decorations (titlebar, etc.) rather than relative to the root window (screen-size background window).

On the X11 platform, the geometry is obtained from the X server, so reflects the latest position of *window*; this may be out-of-sync with the position of *window* delivered in the most-recently-processed **Gnome::Gdk3::EventConfigure**. `get_position()` in contrast gets the position from the most recent configure event.

Note: If *window* is not a toplevel, it is much better to call `get_position()`, `get_width()` and `get_height()` instead, because it avoids the roundtrip to the X server and because these functions support the full 32-bit coordinate space, whereas `get_geometry()` is restricted to the 16-bit coordinates of X11.

    method get-geometry ( --> List )

The List returns

  * Int; X coordinate of window (relative to its parent)

  * Int; Y coordinate of window (relative to its parent)

  * Int; width of window

  * Int; height of window

get-group
---------

Returns the group leader window for *window*. See `set_group()`.

Returns: the group leader window for *window*

    method get-group ( --> N-GObject )

get-height
----------

Returns the height of the given *window*.

On the X11 platform the returned size is the size reported in the most-recently-processed configure event, rather than the current size on the X server.

Returns: The height of *window*

    method get-height ( --> Int )

get-modal-hint
--------------

Determines whether or not the window manager is hinted that *window* has modal behaviour.

Returns: whether or not the window has the modal hint set.

    method get-modal-hint ( --> Bool )

get-origin
----------

Obtains the position of a window in root window coordinates. (Compare with `get_position()` and `get_geometry()` which return the position of a window relative to its parent window.)

Returns: not meaningful, ignore

    method get-origin ( --> Int )

  * $x; return location for X coordinate

  * $y; return location for Y coordinate

get-parent
----------

Obtains the parent of *window*, as known to GDK. Does not query the X server; thus this returns the parent as passed to `new()`, not the actual parent. This should never matter unless you’re using Xlib calls mixed with GDK calls on the X11 platform. It may also matter for toplevel windows, because the window manager may choose to reparent them.

Note that you should use `get_effective_parent()` when writing generic code that walks up a window hierarchy, because `get_parent()` will most likely not do what you expect if there are offscreen windows in the hierarchy.

Returns: parent of *window*

    method get-parent ( --> N-GObject )

get-pass-through
----------------

Returns whether input to the window is passed through to the window below.

See `set_pass_through()` for details

    method get-pass-through ( --> Bool )

get-position
------------

Obtains the position of the window as reported in the most-recently-processed **Gnome::Gdk3::EventConfigure**. Contrast with `get_geometry()` which queries the X server for the current window position, regardless of which events have been received or processed.

The position coordinates are relative to the window’s parent window.

    method get-position ( )

  * $x; X coordinate of window

  * $y; Y coordinate of window

get-root-coords
---------------

Obtains the position of a window position in root window coordinates. This is similar to `get_origin()` but allows you to pass in any position in the window, not just the origin.

    method get-root-coords ( Int() $x, Int() $y )

  * $x; X coordinate in window

  * $y; Y coordinate in window

  * $root_x; return location for X coordinate

  * $root_y; return location for Y coordinate

get-root-origin
---------------

Obtains the top-left corner of the window manager frame in root window coordinates.

    method get-root-origin ( )

  * $x; return location for X position of window frame

  * $y; return location for Y position of window frame

get-scale-factor
----------------

Returns the internal scale factor that maps from window coordiantes to the actual device pixels. On traditional systems this is 1, but on very high density outputs this can be a higher value (often 2).

A higher value means that drawing is automatically scaled up to a higher resolution, so any code doing drawing will automatically look nicer. However, if you are supplying pixel-based data the scale value can be used to determine whether to use a pixel resource with higher resolution data.

The scale of a window may change during runtime, if this happens a configure event will be sent to the toplevel window.

Returns: the scale factor

    method get-scale-factor ( --> Int )

get-screen
----------

Gets the **Gnome::Gdk3::Screen** associated with a **Gnome::Gdk3::Window**.

Returns: the **Gnome::Gdk3::Screen** associated with *window*

    method get-screen ( --> N-GObject )

get-state
---------

Gets the bitwise OR of the currently active window state flags, from the **Gnome::Gdk3::WindowState** enumeration.

Returns: window state bitfield

    method get-state ( --> GdkWindowState )

get-support-multidevice
-----------------------

Returns `True` if the window is aware of the existence of multiple devices.

Returns: `True` if the window handles multidevice features.

    method get-support-multidevice ( --> Bool )

get-toplevel
------------

Gets the toplevel window that’s an ancestor of *window*.

Any window type but `GDK_WINDOW_CHILD` is considered a toplevel window, as is a `GDK_WINDOW_CHILD` window that has a root window as parent.

Note that you should use `get_effective_toplevel()` when you want to get to a window’s toplevel as seen on screen, because `get_toplevel()` will most likely not do what you expect if there are offscreen windows in the hierarchy.

Returns: the toplevel window containing *window*

    method get-toplevel ( --> N-GObject )

get-type-hint
-------------

This function returns the type hint set for a window.

Returns: The type hint set for *window*

    method get-type-hint ( --> GdkWindowTypeHint )

get-user-data
-------------

Retrieves the user data for *window*, which is normally the widget that *window* belongs to. See `set_user_data()`.

    method get-user-data ( Pointer $data )

  * $data; return location for user data

get-visual
----------

Gets the **Gnome::Gdk3::Visual** describing the pixel format of *window*.

Returns: a native **Gnome::Gdk3::Visual**

    method get-visual ( --> N-GObject )

get-width
---------

Returns the width of the given *window*.

On the X11 platform the returned size is the size reported in the most-recently-processed configure event, rather than the current size on the X server.

Returns: The width of *window*

    method get-width ( --> Int )

get-window-type
---------------

Gets the type of the window. See **Gnome::Gdk3::WindowType**.

Returns: type of window

    method get-window-type ( --> GdkWindowType )

has-native
----------

Checks whether the window has a native window or not. Note that you can use `ensure_native()` if a native window is needed.

Returns: `True` if the *window* has a native window, `False` otherwise.

    method has-native ( --> Bool )

hide
----

For toplevel windows, withdraws them, so they will no longer be known to the window manager; for all windows, unmaps them, so they won’t be displayed. Normally done automatically as part of `gtk_widget_hide()`.

    method hide ( )

iconify
-------

Asks to iconify (minimize) *window*. The window manager may choose to ignore the request, but normally will honor it. Using `gtk_window_iconify()` is preferred, if you have a **Gnome::Gdk3::Window** widget.

This function only makes sense when *window* is a toplevel window.

    method iconify ( )

invalidate-rect
---------------

A convenience wrapper around `invalidate_region()` which invalidates a rectangular region. See `invalidate_region()` for details.

    method invalidate-rect ( N-GObject() $rect, Bool $invalidate_children )

  * $rect; rectangle to invalidate or `undefined` to invalidate the whole window

  * $invalidate_children; whether to also invalidate child windows

is-destroyed
------------

Check to see if a window is destroyed..

Returns: `True` if the window is destroyed

    method is-destroyed ( --> Bool )

is-input-only
-------------

Determines whether or not the window is an input only window.

Returns: `True` if *window* is input only

    method is-input-only ( --> Bool )

is-shaped
---------

Determines whether or not the window is shaped.

Returns: `True` if *window* is shaped

    method is-shaped ( --> Bool )

is-viewable
-----------

Check if the window and all ancestors of the window are mapped. (This is not necessarily "viewable" in the X sense, since we only check as far as we have GDK window parents, not to the root window.)

Returns: `True` if the window is viewable

    method is-viewable ( --> Bool )

is-visible
----------

Checks whether the window has been mapped (with `show()` or `show_unraised()`).

Returns: `True` if the window is mapped

    method is-visible ( --> Bool )

lower
-----

Lowers *window* to the bottom of the Z-order (stacking order), so that other windows with the same parent window appear above *window*. This is true whether or not the other windows are visible.

If *window* is a toplevel, the window manager may choose to deny the request to move the window in the Z-order, `lower()` only requests the restack, does not guarantee it.

Note that `show()` raises the window again, so don’t call this function before `show()`. (Try `show_unraised()`.)

    method lower ( )

mark-paint-from-clip
--------------------

If you call this during a paint (e.g. between `begin_paint_region()` and `end_paint()` then GDK will mark the current clip region of the window as being drawn. This is required when mixing GL rendering via `gdk_cairo_draw_from_gl()` and cairo rendering, as otherwise GDK has no way of knowing when something paints over the GL-drawn regions.

This is typically called automatically by GTK+ and you don't need to care about this.

    method mark-paint-from-clip ( cairo_t $cr )

  * $cr; a **cairo_t**

maximize
--------

Maximizes the window. If the window was already maximized, then this function does nothing.

On X11, asks the window manager to maximize *window*, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “maximized”; so you can’t rely on the maximization actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

On Windows, reliably maximizes the window.

    method maximize ( )

merge-child-input-shapes
------------------------

Merges the input shape masks for any child windows into the input shape mask for *window*. i.e. the union of all input masks for *window* and its children will become the new input mask for *window*. See `input_shape_combine_region()`.

This function is distinct from `set_child_input_shapes()` because it includes *window*’s input shape mask in the set of shapes to be merged.

    method merge-child-input-shapes ( )

merge-child-shapes
------------------

Merges the shape masks for any child windows into the shape mask for *window*. i.e. the union of all masks for *window* and its children will become the new mask for *window*. See `shape_combine_region()`.

This function is distinct from `set_child_shapes()` because it includes *window*’s shape mask in the set of shapes to be merged.

    method merge-child-shapes ( )

move
----

Repositions a window relative to its parent window. For toplevel windows, window managers may ignore or modify the move; you should probably use `gtk_window_move()` on a **Gnome::Gdk3::Window** widget anyway, instead of using GDK functions. For child windows, the move will reliably succeed.

If you’re also planning to resize the window, use `move_resize()` to both move and resize simultaneously, for a nicer visual effect.

    method move ( Int() $x, Int() $y )

  * $x; X coordinate relative to window’s parent

  * $y; Y coordinate relative to window’s parent

move-resize
-----------

Equivalent to calling `move()` and `resize()`, except that both operations are performed at once, avoiding strange visual effects. (i.e. the user may be able to see the window first move, then resize, if you don’t use `move_resize()`.)

    method move-resize ( Int() $x, Int() $y, Int() $width, Int() $height )

  * $x; new X position relative to window’s parent

  * $y; new Y position relative to window’s parent

  * $width; new width

  * $height; new height

move-to-rect
------------

Moves *window* to *rect*, aligning their anchor points.

*rect* is relative to the top-left corner of the window that *window* is transient for. *rect_anchor* and *window_anchor* determine anchor points on *rect* and *window* to pin together. *rect*'s anchor point can optionally be offset by *rect_anchor_dx* and *rect_anchor_dy*, which is equivalent to offsetting the position of *window*.

*anchor_hints* determines how *window* will be moved if the anchor points cause it to move off-screen. For example, `GDK_ANCHOR_FLIP_X` will replace `GDK_GRAVITY_NORTH_WEST` with `GDK_GRAVITY_NORTH_EAST` and vice versa if *window* extends beyond the left or right edges of the monitor.

Connect to the *moved-to-rect* signal to find out how it was actually positioned.

    method move-to-rect (
      N-GdkRectangle $rect, GdkGravity $rect_anchor,
      GdkGravity $window_anchor, GdkAnchorHints $anchor_hints,
      Int() $rect_anchor_dx, Int() $rect_anchor_dy
    )

  * $rect; the destination rectangle to align *window* with

  * $rect_anchor; the point on *rect* to align with *window*'s anchor point

  * $window_anchor; the point on *window* to align with *rect*'s anchor point

  * $anchor_hints; positioning hints to use when limited on space

  * $rect_anchor_dx; horizontal offset to shift *window*, i.e. *rect*'s anchor point

  * $rect_anchor_dy; vertical offset to shift *window*, i.e. *rect*'s anchor point

offscreen-window-get-embedder
-----------------------------

Gets the window that window is embedded in.

    method gdk-offscreen-window-get-embedder ( --> N-GObject )

offscreen-window-get-surface
----------------------------

Gets the offscreen surface that an offscreen window renders into. If you need to keep this around over window resizes, you need to add a reference to it.

    method offscreen-window-get-surface ( --> cairo_surface_t )

offscreen-window-set-embedder
-----------------------------

Sets window to be embedded in embedder.

To fully embed an offscreen window, in addition to calling this function, it is also necessary to handle the “pick-embedded-child” signal on the embedder and the “to-embedder” and “from-embedder” signals on window .

    method offscreen-window-set-embedder ( N-GObject() $embedder )

  * $embedder;

peek-children
-------------

Like `get_children()`, but does not copy the list of children, so the list does not need to be freed.

Returns: (element-type GdkWindow): a reference to the list of child windows in *window*

    method peek-children ( --> N-GList )

raise
-----

Raises *window* to the top of the Z-order (stacking order), so that other windows with the same parent window appear below *window*. This is true whether or not the windows are visible.

If *window* is a toplevel, the window manager may choose to deny the request to move the window in the Z-order, `raise()` only requests the restack, does not guarantee it.

    method raise ( )

register-dnd
------------

Registers a window as a potential drop destination.

    method register-dnd ( )

reparent
--------

Reparents *window* into the given *new_parent*. The window being reparented will be unmapped as a side effect.

    method reparent ( N-GObject() $new_parent, Int() $x, Int() $y )

  * $new_parent; new parent to move *window* into

  * $x; X location inside the new parent

  * $y; Y location inside the new parent

resize
------

Resizes *window*; for toplevel windows, asks the window manager to resize the window. The window manager may not allow the resize. When using GTK+, use `gtk_window_resize()` instead of this low-level GDK function.

Windows may not be resized below 1x1.

If you’re also planning to move the window, use `move_resize()` to both move and resize simultaneously, for a nicer visual effect.

    method resize ( Int() $width, Int() $height )

  * $width; new width of the window

  * $height; new height of the window

restack
-------

Changes the position of *window* in the Z-order (stacking order), so that it is above *sibling* (if *above* is `True`) or below *sibling* (if *above* is `False`).

If *sibling* is `undefined`, then this either raises (if *above* is `True`) or lowers the window.

If *window* is a toplevel, the window manager may choose to deny the request to move the window in the Z-order, `restack()` only requests the restack, does not guarantee it.

    method restack ( N-GObject() $sibling, Bool $above )

  * $sibling; a **Gnome::Gdk3::Window** that is a sibling of *window*, or `undefined`

  * $above; a boolean

scroll
------

Scroll the contents of *window*, both pixels and children, by the given amount. *window* itself does not move. Portions of the window that the scroll operation brings in from offscreen areas are invalidated. The invalidated region may be bigger than what would strictly be necessary.

For X11, a minimum area will be invalidated if the window has no subwindows, or if the edges of the window’s parent do not extend beyond the edges of the window. In other cases, a multi-step process is used to scroll the window which may produce temporary visual artifacts and unnecessary invalidations.

    method scroll ( Int() $dx, Int() $dy )

  * $dx; Amount to scroll in the X direction

  * $dy; Amount to scroll in the Y direction

set-accept-focus
----------------

Setting *accept_focus* to `False` hints the desktop environment that the window doesn’t want to receive input focus.

On X, it is the responsibility of the window manager to interpret this hint. ICCCM-compliant window manager usually respect it.

    method set-accept-focus ( Bool $accept_focus )

  * $accept_focus; `True` if the window should receive input focus

set-child-input-shapes
----------------------

Sets the input shape mask of *window* to the union of input shape masks for all children of *window*, ignoring the input shape mask of *window* itself. Contrast with `merge_child_input_shapes()` which includes the input shape mask of *window* in the masks to be merged.

    method set-child-input-shapes ( )

set-child-shapes
----------------

Sets the shape mask of *window* to the union of shape masks for all children of *window*, ignoring the shape mask of *window* itself. Contrast with `merge_child_shapes()` which includes the shape mask of *window* in the masks to be merged.

    method set-child-shapes ( )

set-decorations
---------------

“Decorations” are the features the window manager adds to a toplevel **Gnome::Gdk3::Window**. This function sets the traditional Motif window manager hints that tell the window manager which decorations you would like your window to have. Usually you should use `gtk_window_set_decorated()` on a **Gnome::Gdk3::Window** instead of using the GDK function directly.

The *decorations* argument is the logical OR of the fields in the **GdkWMDecoration** enumeration. If **GDK_DECOR_ALL** is included in the mask, the other bits indicate which decorations should be turned off. If **GDK_DECOR_ALL** is not included, then the other bits indicate which decorations should be turned on.

Most window managers honor a decorations hint of 0 to disable all decorations, but very few honor all possible combinations of bits.

    method set-decorations ( GdkWMDecoration $decorations )

  * $decorations; decoration hint mask

set-device-events
-----------------

Sets the event mask for a given device (Normally a floating device, not attached to any visible pointer) to *window*. For example, an event mask including **Gnome::Gdk3::DK_BUTTON_PRESS_MASK** means the window should report button press events. The event mask is the bitwise OR of values from the **Gnome::Gdk3::EventMask** enumeration.

See the [input handling overview][event-masks] for details.

    method set-device-events ( N-GObject() $device, GdkEventMask $event_mask )

  * $device; **Gnome::Gdk3::Device** to enable events for.

  * $event_mask; event mask for *window*

set-event-compression
---------------------

Determines whether or not extra unprocessed motion events in the event queue can be discarded. If `True` only the most recent event will be delivered.

Some types of applications, e.g. paint programs, need to see all motion events and will benefit from turning off event compression.

By default, event compression is enabled.

    method set-event-compression ( Bool $event_compression )

  * $event_compression; `True` if motion events should be compressed

set-events
----------

The event mask for a window determines which events will be reported for that window from all master input devices. For example, an event mask including **Gnome::Gdk3::DK_BUTTON_PRESS_MASK** means the window should report button press events. The event mask is the bitwise OR of values from the **Gnome::Gdk3::EventMask** enumeration.

See the [input handling overview][event-masks] for details.

    method set-events ( GdkEventMask $event_mask )

  * $event_mask; event mask for *window*

set-focus-on-map
----------------

Setting *focus_on_map* to `False` hints the desktop environment that the window doesn’t want to receive input focus when it is mapped. focus_on_map should be turned off for windows that aren’t triggered interactively (such as popups from network activity).

On X, it is the responsibility of the window manager to interpret this hint. Window managers following the freedesktop.org window manager extension specification should respect it.

    method set-focus-on-map ( Bool $focus_on_map )

  * $focus_on_map; `True` if the window should receive input focus when mapped

set-fullscreen-mode
-------------------

Specifies whether the *window* should span over all monitors (in a multi-head setup) or only the current monitor when in fullscreen mode.

The *mode* argument is from the **Gnome::Gdk3::FullscreenMode** enumeration. If **Gnome::Gdk3::DK_FULLSCREEN_ON_ALL_MONITORS** is specified, the fullscreen *window* will span over all monitors from the **Gnome::Gdk3::Screen**.

On X11, searches through the list of monitors from the **Gnome::Gdk3::Screen** the ones which delimit the 4 edges of the entire **Gnome::Gdk3::Screen** and will ask the window manager to span the *window* over these monitors.

If the XINERAMA extension is not available or not usable, this function has no effect.

Not all window managers support this, so you can’t rely on the fullscreen window to span over the multiple monitors when **Gnome::Gdk3::DK_FULLSCREEN_ON_ALL_MONITORS** is specified.

    method set-fullscreen-mode ( GdkFullscreenMode $mode )

  * $mode; fullscreen mode

set-geometry-hints
------------------

Sets the geometry hints for *window*. Hints flagged in *geom_mask* are set, hints not flagged in *geom_mask* are unset. To unset all hints, use a *geom_mask* of 0 and a *geometry* of `undefined`.

This function provides hints to the windowing system about acceptable sizes for a toplevel window. The purpose of this is to constrain user resizing, but the windowing system will typically (but is not required to) also constrain the current size of the window to the provided values and constrain programatic resizing via `resize()` or `move_resize()`.

Note that on X11, this effect has no effect on windows of type `GDK_WINDOW_TEMP` or windows where override redirect has been turned on via `set_override_redirect()` since these windows are not resizable by the user.

Since you can’t count on the windowing system doing the constraints for programmatic resizes, you should generally call `constrain_size()` yourself to determine appropriate sizes.

    method set-geometry-hints (
      N-GdkGeometry $geometry, GdkWindowHints $geom_mask
    )

  * $geometry; geometry hints

  * $geom_mask; bitmask indicating fields of *geometry* to pay attention to

set-group
---------

Sets the group leader window for *window*. By default, GDK sets the group leader for all toplevel windows to a global window implicitly created by GDK. With this function you can override this default.

The group leader window allows the window manager to distinguish all windows that belong to a single application. It may for example allow users to minimize/unminimize all windows belonging to an application at once. You should only set a non-default group window if your application pretends to be multiple applications.

    method set-group ( N-GObject() $leader )

  * $leader; group leader window, or `undefined` to restore the default group leader window

set-icon-list
-------------

Sets a list of icons for the window. One of these will be used to represent the window when it has been iconified. The icon is usually shown in an icon box or some sort of task bar. Which icon size is shown depends on the window manager. The window manager can scale the icon but setting several size icons can give better image quality since the window manager may only need to scale the icon by a small amount or not at all.

Note that some platforms don't support window icons.

    method set-icon-list ( N-GList $pixbufs )

  * $pixbufs; (element-type GdkPixbuf): A list of pixbufs, of different sizes.

set-icon-name
-------------

Windows may have a name used while minimized, distinct from the name they display in their titlebar. Most of the time this is a bad idea from a user interface standpoint. But you can set such a name with this function, if you like.

After calling this with a non-`undefined` *name*, calls to `set_title()` will not update the icon title.

Using `undefined` for *name* unsets the icon title; further calls to `set_title()` will again update the icon title as well.

Note that some platforms don't support window icons.

    method set-icon-name ( Str $name )

  * $name; name of window while iconified (minimized)

set-keep-above
--------------

Set if *window* must be kept above other windows. If the window was already above, then this function does nothing.

On X11, asks the window manager to keep *window* above, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “keep above”; so you can’t rely on the window being kept above. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

    method set-keep-above ( Bool $setting )

  * $setting; whether to keep *window* above other windows

set-keep-below
--------------

Set if *window* must be kept below other windows. If the window was already below, then this function does nothing.

On X11, asks the window manager to keep *window* below, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “keep below”; so you can’t rely on the window being kept below. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

    method set-keep-below ( Bool $setting )

  * $setting; whether to keep *window* below other windows

set-modal-hint
--------------

The application can use this hint to tell the window manager that a certain window has modal behaviour. The window manager can use this information to handle modal windows in a special way.

You should only use this on windows for which you have previously called `set_transient_for()`

    method set-modal-hint ( Bool $modal )

  * $modal; `True` if the window is modal, `False` otherwise.

set-opacity
-----------

Set *window* to render as partially transparent, with opacity 0 being fully transparent and 1 fully opaque. (Values of the opacity parameter are clamped to the [0,1] range.)

For toplevel windows this depends on support from the windowing system that may not always be there. For instance, On X11, this works only on X screens with a compositing manager running. On Wayland, there is no per-window opacity value that the compositor would apply. Instead, use `set_opaque_region (window, NULL)` to tell the compositor that the entire window is (potentially) non-opaque, and draw your content with alpha, or use `gtk_widget_set_opacity()` to set an overall opacity for your widgets.

For child windows this function only works for non-native windows.

For setting up per-pixel alpha topelevels, see `gdk_screen_get_rgba_visual()`, and for non-toplevels, see `set_composited()`.

Support for non-toplevel windows was added in 3.8.

    method set-opacity ( Num() $opacity )

  * $opacity; opacity

set-override-redirect
---------------------

An override redirect window is not under the control of the window manager. This means it won’t have a titlebar, won’t be minimizable, etc. - it will be entirely under the control of the application. The window manager can’t see the override redirect window at all.

Override redirect should only be used for short-lived temporary windows, such as popup menus. **Gnome::Gdk3::Menu** uses an override redirect window in its implementation, for example.

    method set-override-redirect ( Bool $override_redirect )

  * $override_redirect; `True` if window should be override redirect

set-pass-through
----------------

Sets whether input to the window is passed through to the window below.

The default value of this is `False`, which means that pointer events that happen inside the window are send first to the window, but if the event is not selected by the event mask then the event is sent to the parent window, and so on up the hierarchy.

If *pass_through* is `True` then such pointer events happen as if the window wasn't there at all, and thus will be sent first to any windows below *window*. This is useful if the window is used in a transparent fashion. In the terminology of the web this would be called "pointer-events: none".

Note that a window with *pass_through* `True` can still have a subwindow without pass through, so you can get events on a subset of a window. And in that cases you would get the in-between related events such as the pointer enter/leave events on its way to the destination window.

    method set-pass-through ( Bool $pass_through )

  * $pass_through; a boolean

set-role
--------

When using GTK+, typically you should use `gtk_window_set_role()` instead of this low-level function.

The window manager and session manager use a window’s role to distinguish it from other kinds of window in the same application. When an application is restarted after being saved in a previous session, all windows with the same title and role are treated as interchangeable. So if you have two windows with the same title that should be distinguished for session management purposes, you should set the role on those windows. It doesn’t matter what string you use for the role, as long as you have a different role for each non-interchangeable kind of window.

    method set-role ( Str $role )

  * $role; a string indicating its role

set-shadow-width
----------------

Newer GTK+ windows using client-side decorations use extra geometry around their frames for effects like shadows and invisible borders. Window managers that want to maximize windows or snap to edges need to know where the extents of the actual frame lie, so that users don’t feel like windows are snapping against random invisible edges.

Note that this property is automatically updated by GTK+, so this function should only be used by applications which do not use GTK+ to create toplevel windows.

    method set-shadow-width (
      Int() $left, Int() $right, Int() $top, Int() $bottom
    )

  * $left; The left extent

  * $right; The right extent

  * $top; The top extent

  * $bottom; The bottom extent

set-skip-pager-hint
-------------------

Toggles whether a window should appear in a pager (workspace switcher, or other desktop utility program that displays a small thumbnail representation of the windows on the desktop). If a window’s semantic type as specified with `set_type_hint()` already fully describes the window, this function should not be called in addition, instead you should allow the window to be treated according to standard policy for its semantic type.

    method set-skip-pager-hint ( Bool $skips_pager )

  * $skips_pager; `True` to skip the pager

set-skip-taskbar-hint
---------------------

Toggles whether a window should appear in a task list or window list. If a window’s semantic type as specified with `set_type_hint()` already fully describes the window, this function should not be called in addition, instead you should allow the window to be treated according to standard policy for its semantic type.

    method set-skip-taskbar-hint ( Bool $skips_taskbar )

  * $skips_taskbar; `True` to skip the taskbar

set-source-events
-----------------

Sets the event mask for any floating device (i.e. not attached to any visible pointer) that has the source defined as *source*. This event mask will be applied both to currently existing, newly added devices after this call, and devices being attached/detached.

    method set-source-events ( GdkInputSource $source, GdkEventMask $event_mask )

  * $source; a **Gnome::Gdk3::InputSource** to define the source class.

  * $event_mask; event mask for *window*

set-startup-id
--------------

When using GTK+, typically you should use `gtk_window_set_startup_id()` instead of this low-level function.

    method set-startup-id ( Str $startup_id )

  * $startup_id; a string with startup-notification identifier

set-support-multidevice
-----------------------

This function will enable multidevice features in *window*.

Multidevice aware windows will need to handle properly multiple, per device enter/leave events, device grabs and grab ownerships.

    method set-support-multidevice ( Bool $support_multidevice )

  * $support_multidevice; `True` to enable multidevice support in *window*.

set-title
---------

Sets the title of a toplevel window, to be displayed in the titlebar. If you haven’t explicitly set the icon name for the window (using `set_icon_name()`), the icon name will be set to *title* as well. *title* must be in UTF-8 encoding (as with all user-readable strings in GDK/GTK+). *title* may not be `undefined`.

    method set-title ( Str $title )

  * $title; title of *window*

set-transient-for
-----------------

Indicates to the window manager that *window* is a transient dialog associated with the application window *parent*. This allows the window manager to do things like center *window* on *parent* and keep *window* above *parent*.

See `gtk_window_set_transient_for()` if you’re using **Gnome::Gdk3::Window** or **Gnome::Gdk3::Dialog**.

    method set-transient-for ( N-GObject() $parent )

  * $parent; another toplevel **Gnome::Gdk3::Window**

set-type-hint
-------------

The application can use this call to provide a hint to the window manager about the functionality of a window. The window manager can use this information when determining the decoration and behaviour of the window.

The hint must be set before the window is mapped.

    method set-type-hint ( GdkWindowTypeHint $hint )

  * $hint; A hint of the function this window will have

set-urgency-hint
----------------

Toggles whether a window needs the user's urgent attention.

    method set-urgency-hint ( Bool $urgent )

  * $urgent; `True` if the window is urgent

set-user-data
-------------

For most purposes this function is deprecated in favor of `g_object_set_data()`. However, for historical reasons GTK+ stores the **Gnome::Gdk3::Widget** that owns a **Gnome::Gdk3::Window** as user data on the **Gnome::Gdk3::Window**. So, custom widget implementations should use this function for that. If GTK+ receives an event for a **Gnome::Gdk3::Window**, and the user data for the window is non-`undefined`, GTK+ will assume the user data is a **Gnome::Gdk3::Widget**, and forward the event to that widget.

    method set-user-data ( Pointer $user_data )

  * $user_data; (type GObject.Object): user data

show
----

Like `show_unraised()`, but also raises the window to the top of the window stack (moves the window to the front of the Z-order).

This function maps a window so it’s visible onscreen. Its opposite is `hide()`.

When implementing a **Gnome::Gdk3::Widget**, you should call this function on the widget's **Gnome::Gdk3::Window** as part of the “map” method.

    method show ( )

show-unraised
-------------

Shows a **Gnome::Gdk3::Window** onscreen, but does not modify its stacking order. In contrast, `show()` will raise the window to the top of the window stack.

On the X11 platform, in Xlib terms, this function calls `XMapWindow()` (it also updates some internal GDK state, which means that you can’t really use `XMapWindow()` directly on a GDK window).

    method show-unraised ( )

show-window-menu
----------------

Asks the windowing system to show the window menu. The window menu is the menu shown when right-clicking the titlebar on traditional windows managed by the window manager. This is useful for windows using client-side decorations, activating it with a right-click on the window decorations.

Returns: `True` if the window menu was shown and `False` otherwise.

    method show-window-menu ( GdkEvent $event --> Bool )

  * $event; a **Gnome::Gdk3::Event** to show the menu for

stick
-----

“Pins” a window such that it’s on all workspaces and does not scroll with viewports, for window managers that have scrollable viewports. (When using **Gnome::Gdk3::Window**, `gtk_window_stick()` may be more useful.)

On the X11 platform, this function depends on window manager support, so may have no effect with many window managers. However, GDK will do the best it can to convince the window manager to stick the window. For window managers that don’t support this operation, there’s nothing you can do to force it to happen.

    method stick ( )

test-render-sync
----------------

Retrieves a pixel from window to force the windowing system to carry out any pending rendering commands.

This function is intended to be used to synchronize with rendering pipelines, to benchmark windowing system rendering operations.

    method test-render-sync ( )

test-simulate-button
--------------------

This function is intended to be used in GTK+ test programs. It will warp the mouse pointer to the given (x ,y ) coordinates within window and simulate a button press or release event. Because the mouse pointer needs to be warped to the target location, use of this function outside of test programs that run in their own virtual windowing system (e.g. Xvfb) is not recommended.

Also, gdk_test_simulate_button() is a fairly low level function, for most testing purposes, gtk_test_widget_click() is the right function to call which will generate a button press event followed by its accompanying button release event.

    method test-simulate-button (
      Int() $x, Int() $y, UInt() $button,
      GdkModifierType modifiers, GdkEventType $button_pressrelease
      --> Bool
    )

  * $x; x coordinate within window for the button event

  * $y; y coordinate within window for the button event

  * $button; Number of the pointer button for the event, usually 1, 2 or 3

  * modifiers; Keyboard modifiers the event is setup with

  * $button_pressrelease; either GDK_BUTTON_PRESS or GDK_BUTTON_RELEASE

test-simulate-key
-----------------

This function is intended to be used in GTK+ test programs. If (x ,y ) are > (-1,-1), it will warp the mouse pointer to the given (x ,y ) coordinates within window and simulate a key press or release event.

When the mouse pointer is warped to the target location, use of this function outside of test programs that run in their own virtual windowing system (e.g. Xvfb) is not recommended. If (x ,y ) are passed as (-1,-1), the mouse pointer will not be warped and window origin will be used as mouse pointer location for the event.

Also, gdk_test_simulate_key() is a fairly low level function, for most testing purposes, gtk_test_widget_send_key() is the right function to call which will generate a key press event followed by its accompanying key release event.

    method test-simulate-key (
      Int() $x, Int() $y, UInt $keyval,
      GdkModifierType $modifiers, GdkEventType $key_pressrelease
      --> Bool
    )

  * $x; x coordinate within window for the key event

  * $y; y coordinate within window for the key event

  * $keyval; A GDK keyboard value from **Gnome::Gdk3::Keysyms**

  * $modifiers; Keyboard modifiers the event is setup with

  * $key_pressrelease; either GDK_KEY_PRESS or GDK_KEY_RELEASE

thaw-updates
------------

Thaws a window frozen with `freeze_updates()`.

    method thaw-updates ( )

unfullscreen
------------

Moves the window out of fullscreen mode. If the window was not fullscreen, does nothing.

On X11, asks the window manager to move *window* out of the fullscreen state, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “fullscreen”; so you can’t rely on the unfullscreenification actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

    method unfullscreen ( )

unmaximize
----------

Unmaximizes the window. If the window wasn’t maximized, then this function does nothing.

On X11, asks the window manager to unmaximize *window*, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “maximized”; so you can’t rely on the unmaximization actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

On Windows, reliably unmaximizes the window.

    method unmaximize ( )

unstick
-------

Reverse operation for `stick()`; see `stick()`, and `gtk_window_unstick()`.

    method unstick ( )

withdraw
--------

Withdraws a window (unmaps it and asks the window manager to forget about it). This function is not really useful as `hide()` automatically withdraws toplevel windows before hiding them.

    method withdraw ( )

Signals
=======

There are two ways to connect to a signal. The first option you have is to use `register-signal()` from **Gnome::GObject::Object**. The second option is to use `connect-object()` directly from **Gnome::GObject::Signal**.

First method
------------

The positional arguments of the signal handler are all obligatory as well as their types. The named attributes `:$widget` and user data are optional.

    # handler method
    method mouse-event ( GdkEvent $event, :$widget ) { ... }

    # connect a signal on window object
    my Gnome::Gtk3::Window $w .= new( ... );
    $w.register-signal( self, 'mouse-event', 'button-press-event');

Second method
-------------

    my Gnome::Gtk3::Window $w .= new( ... );
    my Callable $handler = sub (
      N-GObject $native, GdkEvent $event, OpaquePointer $data
    ) {
      ...
    }

    $w.connect-object( 'button-press-event', $handler);

Also here, the types of positional arguments in the signal handler are important. This is because both methods `register-signal()` and `connect-object()` are using the signatures of the handler routines to setup the native call interface.

Supported signals
-----------------

### create-surface

The *create-surface* signal is emitted when an offscreen window needs its surface (re)created, which happens either when the window is first drawn to, or when the window is being resized. The first signal handler that returns a non-`undefined` surface will stop any further signal emission, and its surface will be used.

Note that it is not possible to access the window's previous surface from within any callback of this signal. Calling `gdk_offscreen_window_get_surface()` will lead to a crash.

Returns: the newly created **cairo_surface_t** for the offscreen window

    method handler (
      Int $width,
      Int $height,
      Int :$_handle_id,
      Gnome::GObject::Object :_widget($window),
      *%user-options
      --> Unknown type CAIRO_GOBJECT_TYPE_SURFACE
    );

  * $window; the offscreen window on which the signal is emitted

  * $width; the width of the offscreen surface to create

  * $height; the height of the offscreen surface to create

  * $_handle_id; the registered event handler id

Properties
==========

An example of using a string type property of a **Gnome::Gtk3::Label** object. This is just showing how to set/read a property, not that it is the best way to do it. This is because a) The class initialization often provides some options to set some of the properties and b) the classes provide many methods to modify just those properties. In the case below one can use **new(:label('my text label'))** or **.set-text('my text label')**.

    my Gnome::Gtk3::Label $label .= new;
    my Gnome::GObject::Value $gv .= new(:init(G_TYPE_STRING));
    $label.get-property( 'label', $gv);
    $gv.set-string('my text label');

Supported properties
--------------------

### Cursor: cursor

The mouse pointer for a **Gnome::Gdk3::Window**. See `set_cursor()` and `get_cursor()` for details.

    Widget type: GDK_TYPE_CURSOR

The **Gnome::GObject::Value** type of property *cursor* is `G_TYPE_OBJECT`.

