package IO::Select::SSL;

use 5.006;
use strict;
use warnings;
use base qw(IO::Select);

our $VERSION = '0.02';

# can_read( [ $timeout ] )
# Assume pending SSL handles have priority since no select() syscall is required
sub can_read {
  my $self = shift;
  if (my @ssl = $self->pending_handles) {
    return @ssl;
  }
  return $self->SUPER::can_read(@_);
}

# select ( READ, WRITE, ERROR [, TIMEOUT ] )
# Assume pending SSL read handles have priority since no select() syscall is required
sub select {
  my $class = shift;
  if (my $reader = $_[0]) {
    if (my @ssl = $reader->pending_handles) {
      return (\@ssl, [], []);
    }
  }
  return $class->SUPER::select(@_);
}

# pending_handles
# Return a list of handles that already have pending but unread data in its INPUT buffer.
sub pending_handles {
  my $self = shift;
  my @ssl = ();
  foreach my $handle ($self->handles) {
    if (ref($handle) =~ /::/ and
        $handle->can("pending") and
        $handle->pending) {
      push @ssl, $handle;
    }
  }
  if (@ssl) {
    return @ssl;
  }
  return ();
}

# Autoload methods go after =cut, and are processed by the autosplit program.

1;
__END__

=head1 NAME

IO::Select::SSL - IO::Socket::SSL compatible IO::Select

=head1 SYNOPSIS

  use IO::Select::SSL;
  my $sel = new IO::Select::SSL;


=head1 DESCRIPTION

This module is intended to be a drop-in replacement for IO::Select.
However, the can_read method actually handles the very special IO::Socket::SSL
handles correctly by returning those handles that still have at least some
decrypted characters in the buffer thus won't block when sysread is called.

=head2 METHODS

  can_read ( [ $timeout ] )

Same as IO::Select except immediately returns handles with pending INPUT data.

  select ( READ, WRITE, ERROR [, TIMEOUT ] )

Same as IO::Select except immediately returns handles with pending INPUT data.

  pending_handles

Returns a list of readable handles with pending INPUT data.

=head1 AUTHOR

Rob Brown E<lt>bbb@cpan.orgE<gt>

=head1 SEE ALSO

L<IO::Select>.
L<IO::Socket::SSL>.

=cut
