# $Id: Code.pm,v 1.21 2003/12/27 22:32:19 jeff Exp $

package ExtProc::Code;

use 5.6.1;
use strict;
use warnings;
no strict qw(subs);

require Exporter;
require DynaLoader;

our @ISA = qw(Exporter DynaLoader);
our %EXPORT_TAGS = ( 'all' => [ qw(
	&create_wrapper
        &import_code
        &drop_code
) ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );
our @EXPORT = qw(
);
our $VERSION = '1.99_04';

use ExtProc;
use File::Spec;
use DBI;

# for converting from *supported* PL/SQL datatypes to C datatypes
# should explore integrating this with OTT somehow
my %typemap = (
	'PLS_INTEGER' => {
		'IN'		=> 'int',
		'OUT'		=> 'int *',
		'IN OUT'	=> 'int *',
		'RETURN'	=> 'int',
		'PARAMTYPE'	=> 'int',
		'NULLABLE'	=> 0,
		'VARLENGTH'	=> 0
	},
	'REAL' => {
		'IN'		=> 'float',
		'OUT'		=> 'float *',
		'IN OUT'	=> 'float *',
		'RETURN'	=> 'float',
		'PARAMTYPE'	=> 'float',
		'NULLABLE'	=> 0,
		'VARLENGTH'	=> 0
	},
	'VARCHAR2' => {
		'IN'		=> 'char *',
		'OUT'		=> 'char *',
		'IN OUT'	=> 'char *',
		'RETURN'	=> 'char *',
		'PARAMTYPE'	=> 'string',
		'NULLABLE'	=> 1,
		'VARLENGTH'	=> 1
	},
	'void'	=> {
		'IN'		=> 'void',
		'RETURN'	=> 'void'
	}
);

# prefix for C functions to avoid name clashes with standard C library
our $c_prefix = "EP_";

# create_wrapper(proto)
# create C wrapper function in trusted code directory based on prototype
sub create_wrapper
{
	my $proto = shift;

	# @args holds all information about each argument
	# @args = ( { spec => x, type => x, inout => x, carg => x } )
	my @args;

	# if prototype is valid, write out C wrapper to trusted directory
	# prototype format: name([arg1[,arg2,...]]) [RETURN type]
	if ($proto =~ /^(\w+)\s*\((.*?)\)(?:\s+RETURN\s+([\w*]+))*$/io) {
		my $name = $1;
		my @a = split(/,\s*/, $2);
		my $rettype = (defined $3) ? uc($3) : 'void';
		my $subtype = (defined $3) ?
			'EP_SUBTYPE_FUNCTION' :
			'EP_SUBTYPE_PROCEDURE';
		my $n = 0;

		# convert prototype to C arguments
		foreach (@a) {
			my ($argname, $inout, $type);
			if (/^([\w_]+)\s+IN\s+OUT\s+(.+)$/i) {
				$argname = $1;
				$inout = "IN OUT";
				$type = $2;
			}
			else {
				($argname, $inout, $type) = split(/\s+/);
			}
			$type = uc($type);
			unless (exists $typemap{$type}) {
				die "unsupported datatype: $type";
			}
			$inout = uc($inout);
			my $ctype = $typemap{$type}{$inout};
			$args[$n]{'spec'} = $_;
			$args[$n]{'name'} = $argname;
			$args[$n]{'type'} = $type;
			$args[$n]{'inout'} = $inout;
			my $tmp = "$ctype arg$n";
			if ($typemap{$type}{'NULLABLE'}) {
				$tmp .= ", OCIInd is_null_$n";
			}
			if ($typemap{$type}{'VARLENGTH'}) {
				if ($inout =~ /OUT/) {
					$tmp .= ", sb4 *length_$n, sb4 *maxlen_$n";
				}
				else {
					$tmp .= ", sb4 length_$n";
				}
			}
			$args[$n]{'carg'} = $tmp;
			$n++;
		}
		my $argstr = join(', ', map($_->{'carg'}, @args));
		$argstr = ", $argstr" if $n;

		# result declaration and fatal return statement
		my $return_fatal;
		unless (exists $typemap{$rettype} || $rettype eq 'void') {
			die "unsupported datatype: $rettype";
		}
		my $crettype = $typemap{$rettype}{'RETURN'};
		my $result_dec = "$crettype res;";
		if ($crettype eq 'void') {
			$return_fatal = "return";
			$result_dec = "";
		}
		elsif ($crettype eq 'int' || $crettype eq 'float') {
			$return_fatal = "return(0)";
		}
		else {
			$return_fatal = "return(NULL)";
		}

		# write to file in trusted code directory
		local *CODE;
		my $dir = ExtProc::config('trusted_code_directory');
		open(CODE, '>', File::Spec->catfile($dir, $name . '.c' ))
			or die $!;

		# generate C wrapper source file
		print CODE <<_DONE_;
/* THIS FILE IS AUTOGENERATED -- CHANGES MAY BE LOST */

#ifdef __cplusplus
extern "C" {
#endif
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <oci.h>

/* Perl headers */
#define PERL_NO_GET_CONTEXT
#include <EXTERN.h>
#include <perl.h>

#include "extproc_perl.h"
#ifdef __cplusplus
}
#endif

/* per-session context -- contains all the globals from version 1 */
extern EP_CONTEXT my_context;

_DONE_

		# don't need return indicator for procedures
		if ($rettype eq 'void') {
			print CODE "$crettype $c_prefix$name(OCIExtProcContext *ctx $argstr)\n";
		}
		else {
			print CODE "$crettype $c_prefix$name(OCIExtProcContext *ctx, OCIInd *ret_ind $argstr)\n";
		}

		print CODE <<_DONE_;
{
	int nret;
	short ind;
	SV *sv, *svcache[$#args+1];
	char *fqsub, *tmp;
	EP_CONTEXT *c;
	EP_CODE code;
	STRLEN len;
	$result_dec

	dTHX;

	dSP;

	c = &my_context;

	_ep_init(c, ctx);
_DONE_

		if ($rettype eq 'void') {
			print CODE "\tEP_DEBUGF(c, \"IN (user defined) $c_prefix$name(%p, ...)\", ctx);\n";
		}
		else {
			print CODE "\tEP_DEBUGF(c, \"IN (user defined) $c_prefix$name(%p, %p, ...)\", ctx, ret_ind);\n";
		}

		print CODE <<_DONE_;
	EP_DEBUGF(c, "-- prototype: $proto", ctx);

	c->subtype = $subtype;

	/* start perl interpreter if necessary */
	if (!c->perl) {
		c->perl = pl_startup(c);
		if (!c->perl) {
_DONE_

		if ($rettype ne 'void') {
			print CODE "\t\t\t*ret_ind = OCI_IND_NULL;\n";
		}

		print CODE <<_DONE_;
			ora_exception(c, "interpreter initialization failed");
			$return_fatal;
		}
	}
	EP_DEBUG(c, "RETURN $c_prefix$name");

	SPAGAIN; /* in case we started interpreter after declaring SP */
	ENTER;
	SAVETMPS;
	PUSHMARK(SP);

_DONE_
		foreach my $n (0..$#args) {
			print CODE "\t/* push $args[$n]{'spec'} arg$n onto stack */\n";
			if ($typemap{$args[$n]{'type'}}{'NULLABLE'}) {
				print CODE <<_DONE_;
	if (is_null_$n == OCI_IND_NULL) {
		sv = sv_2mortal(newSVsv(&PL_sv_undef));
	}
	else {
_DONE_
			}
			my $star = ($args[$n]{'inout'} =~ /OUT/) ? "*" : "";
			if ($args[$n]{'carg'} =~ /^char /) {
				print CODE "\tsv = sv_2mortal(newSVpvn(arg$n, ${star}length_$n));\n";
			}
			elsif ($args[$n]{'carg'} =~ /^int /) {
				print CODE "\tsv = sv_2mortal(newSViv(${star}arg$n));\n";
			}
			elsif ($args[$n]{'carg'} =~ /^float /) {
				print CODE "\tsv = sv_2mortal(newSVnv(${star}arg$n));\n";
			}
			else {
				die "unsupported C datatype: $args[$n]{'carg'} (was $args[$n]{'spec'})";
			}

			print CODE <<_DONE_;
	svcache[$n] = sv;
	if (c->tainting) {
		SvTAINTED_on(sv);
	}
_DONE_
			if ($typemap{$args[$n]{'type'}}{'NULLABLE'}) {
				print CODE "\t}\n";
			}

			# IN OUT & OUT types are always passed as references
			if ($args[$n]{'inout'} =~ /OUT/) {
				print CODE "\tXPUSHs(newRV_noinc(sv));\n";
			}
			else {
				print CODE "\tXPUSHs(sv);\n";
			}
		}

		print CODE <<_DONE_;
	PUTBACK;

	fqsub = parse_code(c, &code, "$name");
	EP_DEBUG(c, "RETURN (user defined) $c_prefix$name");
	if (!fqsub) {
_DONE_
			if ($rettype ne 'void') {
				print CODE "\t\t*ret_ind = OCI_IND_NULL;\n";
			}

			print CODE <<_DONE_;
		ora_exception(c, "invalid subroutine");
		$return_fatal;
	}

	EP_DEBUG(c, "-- about to call call_pv()");
	nret = call_pv(fqsub, G_SCALAR|G_EVAL);
	EP_DEBUGF(c, "-- call_pv() returned %d", nret);
	SPAGAIN;
_DONE_

		# copy values to IN OUT and OUT args
		foreach my $n (0..$#args) {
			if ($args[$n]{'inout'} =~ /OUT/) {
				if ($args[$n]{'carg'} =~ /^char /) {
					print CODE <<_DONE_;
	tmp = SvPV(svcache[$n], len);
	if (len > *maxlen_$n) {
		ora_exception(c, "length of arg$n exceeds maximum length for parameter");
		$return_fatal;
	}
	Copy(tmp, arg$n, len, char);
	*length_$n = len;
_DONE_
				}
				elsif ($args[$n]{'carg'} =~ /^int /) {
					print CODE "\t*arg$n = SvIV(svcache[$n]);\n";
				}
				elsif ($args[$n]{'carg'} =~ /^float /) {
					print CODE "\t*arg$n = SvNV(svcache[$n]);\n";
				}
				else {
					die "unsupported C datatype: $args[$n]{'carg'} (was $args[$n]{'spec'})";
				}
			}
		}

		# procedures don't return values
		if ($rettype eq 'void') {
			print CODE <<_DONE_;
	/* clean up stack and return */
	PUTBACK;
	FREETMPS;
	LEAVE;
	return;
}
_DONE_
		}
		# functions do return values
		else {
			print CODE <<_DONE_;

	/* grab return value off the stack */
	sv = POPs;
_DONE_
			if ($crettype =~ /char\s*\*/) {
				print CODE <<_DONE_;
        tmp = SvPV(sv,len);
        New(0, res, len+1, char);
        Copy(tmp, res, len, char);
        res[len] = '\\0';
_DONE_
			}
			elsif ($crettype eq 'int') {
				print CODE <<_DONE_;
	res = SvIV(sv);
_DONE_
			}
			elsif ($crettype eq 'float') {
				print CODE <<_DONE_;
	res = SvNV(sv);
_DONE_
			}
			else {
				die "unknown return type: $crettype";
			}

			print CODE <<_DONE_;

	*ret_ind = SvOK(sv) ? OCI_IND_NOTNULL : OCI_IND_NULL;

	/* clean up stack and return */
	PUTBACK;
	FREETMPS;
	LEAVE;

	return(res);
}
_DONE_
		}
		close(CODE);

		# external procedure spec
		local *SPEC;
		open(SPEC, '>', File::Spec->catfile($dir, $name . '.sql' ))
			or die $!;
		print SPEC "CREATE OR REPLACE ", ($rettype eq 'void') ?
			"PROCEDURE" : "FUNCTION", " $proto\n";
		print SPEC "AS EXTERNAL NAME \"EP_$name\"\n";
		print SPEC "LIBRARY \"PERL_LIB\"\n";
		print SPEC "WITH CONTEXT\n";
		print SPEC "PARAMETERS (\n";
		print SPEC "   CONTEXT";
		($rettype eq 'void') or print SPEC ",\n   RETURN INDICATOR BY REFERENCE\n";
		foreach my $n (0..$#args) {
			my $name = $args[$n]{'name'};
			my $type = $args[$n]{'type'};
			my $inout = $args[$n]{'inout'};
			print SPEC ",\n   $name $typemap{$type}{'PARAMTYPE'}";
			if ($typemap{$type}{'NULLABLE'}) {
				print SPEC ",\n   $name INDICATOR short";
			}
			if ($typemap{$type}{'VARLENGTH'}) {
				print SPEC ",\n   $name LENGTH sb4";
				if ($inout =~ /OUT/) {
					print SPEC ",\n   $name MAXLEN sb4";
				}
			}
		}
		print SPEC "\n);\n/\n";
		close(SPEC);
	}
	else {
		die "invalid prototype";
	}

}

# import_code(name, filename, [proto])
# import code from a file in the trusted code directory and optionally create
# a C wrapper based on the supplied prototype
sub import_code
{
	my ($name, $file, $proto) = @_;

	# DML -- MUST BE CALLED AS A PROCEDURE
	if (!ExtProc::is_procedure) {
		ExtProc::ora_exception('import_code must be called as a procedure!');
		return;
	}

	if ($name eq '' || $file eq '') {
		ExtProc::ora_exception('import_code: empty subroutine name or filename');
		return;
	}

	# untaint arguments, since we're being called from oracle
	if ($name =~ /^([A-z\d\-_]+)$/) {
		$name = $1;
	}
	else {
		ExtProc::ora_exception('illegal characters in subroutine name');
		return;
	}
	if ($file =~ /^([\w\.\-\_]+)$/) {
		$file = $1;
	}
	else {
		ExtProc::ora_exception('illegal characters in filename');
		return;
	}

	# what's our code table and trusted code directory?
	my $table = ExtProc::config('code_table');
	my $dir = ExtProc::config('trusted_code_directory');

	my $path = File::Spec->catfile($dir, $file);
	my $size = (stat($path))[7];
	if ($size > 4000) {
		ExtProc::ora_exception("file too large for import ($size bytes)");
		return;
	}

	# read code from file
	my ($code, $line);
	local *CODE;
	open(CODE, $path) or die "failed to open code file: $!";
	while(defined($line = <CODE>)) {
		$code .= $line;
	}
	close(CODE);

	# delete existing code if it exists
	my $dbh = ExtProc->dbi_connect;
	my $sth = $dbh->prepare("delete from $table where name = ?");
	$sth->execute($name);
	$sth->finish;

	# import code into database
	$sth = $dbh->prepare("insert into $table (name, language, last_modified_user, last_modified_date, code) values(?, 'Perl5', user, SYSDATE, ?)");
	$sth->execute($name, $code);
	$sth->finish;

	# if we have a prototype, create the wrapper
	if (defined($proto)) {
		create_wrapper($proto);
	}
}

# drop_code(name)
# silently remove code from code table
sub drop_code
{
	my $name = shift;

	# DML -- MUST BE CALLED AS A PROCEDURE
	if (!ExtProc::is_procedure) {
		ExtProc::ora_exception('drop_code must be called as a procedure!');
		return;
	}

	# what's our code table?
	my $table = ExtProc::config('code_table');

	my $dbh = ExtProc->dbi_connect;
	my $sth = $dbh->prepare("delete from $table where name = ?");
	$sth->execute($name);
	$sth->finish;
}
