#! /usr/bin/perl -w

use strict;
use warnings;
use Test::More;

use Dir::Self;
use File::Spec;
use File::Temp 'tempdir';

use lib File::Spec->catdir(__DIR__, 'lib');

my ($sha_ok, $bcrypt_ok, $pgp_ok, $whirlpool_ok);

BEGIN {
  $sha_ok    = eval 'require Digest' && eval 'require Digest::SHA;';
  $bcrypt_ok = eval 'require Crypt::Eksblowfish::Bcrypt';
  $pgp_ok    = eval 'require Crypt::OpenPGP';
  $whirlpool_ok = eval 'require Digest; 1' && eval 'require Digest::Whirlpool; 1';
}

my $tests = 5;
$tests += 22 if $sha_ok;
$tests += 6  if $bcrypt_ok;
$tests += 6  if $pgp_ok;
$tests += 7  if $whirlpool_ok;

plan tests => $tests;

#1
use_ok('DigestTest::Schema');

my $tmp = tempdir( CLEANUP => 1 );
my $db_file = File::Spec->catfile($tmp, 'testdb.sqlite');
my $schema = DigestTest::Schema->connect("dbi:SQLite:dbname=${db_file}");
$schema->deploy({}, File::Spec->catdir(__DIR__, 'var'));

my $rs = $schema->resultset('Test');

my $checks = {};
if( $sha_ok ){
  for my $algorithm( qw/SHA-1 SHA-256/){
    my $maker = Digest->new($algorithm);
    my $encodings = $checks->{$algorithm} = {};
    for my $encoding (qw/base64 hex/){
      my $values = $encodings->{$encoding} = {};
      my $encoding_method = $encoding eq 'binary' ? 'digest' :
        ($encoding eq 'hex' ? 'hexdigest' : 'b64digest');
      for my $value (qw/test1 test2/){
        $maker->add($value);
        $values->{$value} = $maker->$encoding_method;
      }
    }
  }
}
if ( $whirlpool_ok ) {
  for my $algorithm( qw/Whirlpool/){
    my $maker = Digest->new($algorithm);
    my $encodings = $checks->{$algorithm} = {};
    for my $encoding (qw/base64 hex/){
      my $values = $encodings->{$encoding} = {};
      my $encoding_method = $encoding eq 'binary' ? 'digest' :
        ($encoding eq 'hex' ? 'hexdigest' : 'b64digest');
      for my $value (qw/test1 test2/){
        $maker->reset()->add($value);
        $values->{$value} = $maker->$encoding_method;
      }
    }
  }
}

my %create_vals = (dummy_col  => 'test1');
if( $sha_ok ){
  $create_vals{$_} = 'test1'
    for(qw/sha1_hex sha1_b64 sha256_hex sha256_b64 sha256_b64_salted/);
}
if( $whirlpool_ok ){
  $create_vals{$_} = 'test1'
    for(qw/whirlpool_hex whirlpool_b64/);
}

if( $bcrypt_ok ){
  $create_vals{$_} = 'test1' for(qw/bcrypt_1 bcrypt_2/);
}

my $row = $rs->create( \%create_vals );

#2
is($row->dummy_col,  'test1','dummy on create');
ok(!$row->can('check_dummy_col'));

#8
if( $bcrypt_ok ){
  is( length($row->bcrypt_1), 60, 'correct length');
  is( length($row->bcrypt_2), 59, 'correct length');

  ok( $row->bcrypt_1_check('test1'));
  ok( $row->bcrypt_2_check('test1'));

  $row->bcrypt_1('test2');
  $row->bcrypt_2('test2');

  ok( $row->bcrypt_1_check('test2'));
  ok( $row->bcrypt_2_check('test2'));
}

#14
if( $sha_ok ) {
  is($row->sha1_hex,   $checks->{'SHA-1'}{hex}{test1},     'hex sha1 on create');
  is($row->sha1_b64,   $checks->{'SHA-1'}{base64}{test1},  'b64 sha1 on create');
  is($row->sha256_hex, $checks->{'SHA-256'}{hex}{test1},   'hex sha256 on create');
  is($row->sha256b64,  $checks->{'SHA-256'}{base64}{test1},'b64 sha256 on create');
  is( length($row->sha256_b64_salted), 57, 'correct salted length');

#   my $salted_check = sub {
#     my $col_v = $_[0]->get_column('sha256_b64_salted');
#     my $target = substr($col_v, 0, 43);
#     my $salt   = substr($col_v, 43);
#     my $maybe = $_[0]->_column_encoders->{'sha256_b64_salted'}->($_[1], $salt);
#     print STDERR "$_[1]\t${salt}\t${maybe}\n";
#     $maybe eq $col_v;
#  };

  #die unless $salted_check->($row, 'test1');

  can_ok($row, qw/check_sha1_hex check_sha1_b64/);
  ok($row->check_sha1_hex('test1'),'Checking hex digest_check_method');
  ok($row->check_sha1_b64('test1'),'Checking b64 digest_check_method');
  ok($row->check_sha256_b64_salted('test1'), 'Checking salted digest_check_method');

  $row->sha1_hex('test2');
  is($row->sha1_hex, $checks->{'SHA-1'}{hex}{test2}, 'Checking accessor');

  $row->update({sha1_b64 => 'test2',  dummy_col => 'test2'});
  is($row->sha1_b64, $checks->{'SHA-1'}{base64}{test2}, 'Checking update');
  is($row->dummy_col,  'test2', 'dummy on update');

  $row->set_column(sha256_hex => 'test2');
  is($row->sha256_hex, $checks->{'SHA-256'}{hex}{test2}, 'Checking set_column');

  $row->sha256b64('test2');
  is($row->sha256b64, $checks->{'SHA-256'}{base64}{test2}, 'custom accessor');

  $row->update;

} else {
  #1
  $row->update({dummy_col => 'test2'});
  is($row->dummy_col,  'test2', 'dummy on update');

}

#4
if( $sha_ok ) {
  my $copy = $row->copy({sha256_b64 => 'test2'});
  is($copy->sha1_hex,   $checks->{'SHA-1'}{hex}{test2},     'hex sha1 on copy');
  is($copy->sha1_b64,   $checks->{'SHA-1'}{base64}{test2},  'b64 sha1 on copy');
  is($copy->sha256_hex, $checks->{'SHA-256'}{hex}{test2},   'hex sha256 on copy');
  is($copy->sha256b64,  $checks->{'SHA-256'}{base64}{test2},'b64 sha256 on copy');
}

# 7
if( $whirlpool_ok ){
  is( $row->whirlpool_hex, $checks->{'Whirlpool'}{hex}{test1}, 'Whirlpool hex');
  is( $row->whirlpool_b64, $checks->{'Whirlpool'}{base64}{test1}, 'Whirlpool b64');

  can_ok( $row, qw/check_whirlpool_hex check_whirlpool_b64/ );
  ok( $row->check_whirlpool_hex('test1'), 'Checking hex digest_check_method for Whirlpool');
  ok( $row->check_whirlpool_b64('test1'), 'Checking b64 digest_check_method for Whirlpool');

  $row->whirlpool_hex('test2');
  is( $row->whirlpool_hex, $checks->{'Whirlpool'}{hex}{test2}, 'Checking accessor (Whirlpool)');

  $row->update({ whirlpool_b64 => 'test2' });
  is( $row->whirlpool_b64, $checks->{'Whirlpool'}{base64}{test2}, 'Checking Update (Whirlpool)');

}

#1
my $new = $rs->new( \%create_vals );
is($new->dummy_col,  'test1', 'dummy on new');

#4
if( $sha_ok ){
  is($new->sha1_hex,   $checks->{'SHA-1'}{hex}{test1},      'hex sha1 on new');
  is($new->sha1_b64,   $checks->{'SHA-1'}{base64}{test1},   'b64 sha1 on new');
  is($new->sha256_hex, $checks->{'SHA-256'}{hex}{test1},    'hex sha256 on new');
  is($new->sha256b64,  $checks->{'SHA-256'}{base64}{test1}, 'b64 sha256 on new');
}

#6
if ( $pgp_ok ) {
  my $row = $rs->create( {
    dummy_col          => 'Dummy Column',
    pgp_col_passphrase => 'Test Encrypted Column with Passphrase',
    pgp_col_key        => 'Test Encrypted Column with Key Exchange',
    pgp_col_key_ps     => 'Test Encrypted Column with Key Exchange + Pass',
  } );

  like($row->pgp_col_passphrase, qr/BEGIN PGP MESSAGE/, 'Passphrase encrypted');
  like($row->pgp_col_key, qr/BEGIN PGP MESSAGE/, 'Key encrypted');
  like($row->pgp_col_key_ps, qr/BEGIN PGP MESSAGE/, 'Key+Passphrase encrypted');

  is(
    $row->decrypt_pgp_passphrase('Secret Words'),
    'Test Encrypted Column with Passphrase',
    'Passphrase decryption/encryption'
  );

  is(
    $row->decrypt_pgp_key,
    'Test Encrypted Column with Key Exchange',
    'Key Exchange decryption/encryption'
  );

  is(
    $row->decrypt_pgp_key_ps('Secret Words'),
    'Test Encrypted Column with Key Exchange + Pass',
    'Secured Key Exchange decryption/encryption'
  );


}

if( $sha_ok ){
    $row->sha1_hex(undef);
    $row->update;
    is($row->sha1_hex, undef, 'Check undef is passed through');
}

#TODO
# -- dies_ok tests when using invalid cyphers and encodings

1;
