# Your text editor is going to love this one! -*- perl -*-
package RISCOS::Text::PrettyPrint;

=head1 NAME

RISCOS::Text::PrettyPrint --perl module that emulates OS_PrettyPrint

=head1 SYNOPSIS

    use RISCOS::Text::PrettyPrint;
    $expand = prettyprint_expand ($sometext, $special);

    @kernel_dict = read_riscosdict ('Resources:$.Resources.Kernel.Dictionary');
    $expand = prettyprint_expand ($sometext, '', @kernel_dict);

=head1 DESCRIPTION

S<RISC OS> provides a call C<OS_PrettyPrint> that both formats text in a
I<pretty> fashion and allows a limited form of text compaction by using a
dictionary. C<prettyprint_expand> does not perform the line wrapping or tab
expansion of C<OS_PrettyPrint> (see L<Text::Wrap> and L<Text::Tabs> if this is
needed) but does perform the dictionary expansion and converts the
C<OS_PrettyPrint> codes for non-breaking space and newline (C<<31E<gt>> and
C<<13E<gt>> respectively) to C<<160E<gt>> and C<<10E<gt>> (a newline).

The second (optional) parameter to C<prettyprint_expand> is a I<special string>
which is used as dictionary entry zero. The remaining parameters are form the
dictionary. If no dictionary is supplied then the default S<RISC OS> dictionary
is used (as in C<OS_PrettyPrint>), which contains text common to the syntax
messages of many modules.

C<read_riscosdict> reads the C<OS_PrettyPrint> dictionary format and converts it
to an array. If passed a string this will be taken as a filename and opened. If
passed a reference to a filehandle this will be read from. (This allows reading
of the default dictionary from the C<DATA> filehadle of this module.

=head1 EXAMPLE

This will print out the default dictionary.

    use RISCOS::Text::PrettyPrint ':DEFAULT', '@default_dict';

    foreach $entry (0 .. $#default_dict)
    {
	print prettyprint_expand (sprintf ("%2d: \c[%c\n", $entry, $entry),
				  '<Special string>');
    }

=head1 AUTHOR

Nicholas Clark <F<nick@unfortu.net>>

=cut

require Exporter;
use Carp;
use strict;
use vars qw (@ISA @EXPORT @EXPORT_OK $VERSION @default_dict);

$VERSION = 0.02;	# Now does open or typeglob
@ISA = qw(Exporter);
@EXPORT = qw(prettyprint_expand read_riscosdict);
@EXPORT_OK = qw(@default_dict);

sub prettyprint_expand ($;$@)
{
    my $text = shift;
    my $special = shift;
    my @dict = @_;
    @dict = @default_dict unless @dict;

    my $max = $#_;

    $dict[0] = defined ($special) ? $special : '';
    # Insert the special string

    while ($text =~ s/\c[(.)/$dict[ord $1]/es)
    {
	next if $max--;
	carp sprintf "Deep recursion on expansion of dictionary entry %d",
		     ord $1;
	return undef;
    }

    $text =~ y/\r\x1F/\n/;	# CR -> NL, <31> to hard space
    $text
}

sub read_riscosdict ($)
{
    my $file = shift;
    local *FILE;

    if (ref($file) ? (ref($file) eq 'GLOB'
		      || UNIVERSAL::isa($file, 'GLOB')
		      || UNIVERSAL::isa($file, 'IO::Handle'))
		   : (ref(\$file) eq 'GLOB'))
    {
        *FILE = $file;
    }
    else
    {
        open FILE, "<$file" or croak "Unable to open '$file': $!";
    }

    my @dict = '';			# First entry is blank
    my ($length, $entry);

    while (1)
    {
	unless (read FILE, $length ,1)
	{
	    warn printf "Could not read length of dictionary entry %d: $!",
			scalar @dict;
	    return;
	}
	$length = ord ($length);	# We've read the length byte already

	unless ($length--)		# so subtract it
	{
	    warn "Not at end of dictionary file" if ($^W && !eof FILE);
	    return @dict;
	}

	unless (read FILE, $entry ,$length)
	{
	    warn printf "Could not read dictionary entry %d: $!", scalar @dict;
	    return;
	}
	chop $entry;			# Remove trailing '\0'
	push @dict, $entry;
    }
}

# Use this as the return value:
@default_dict = read_riscosdict (\*DATA);

# Your text editor is going to love this
# There is no newline after the final two '\0's
__DATA__

Syntax: *   the  
director filing system 	current = to a variable. Other types of value can be assigned with * file 
default  tion 
*Configure  name 	 server number  < - one or more s that matchgiven wildcard  and  relocatable module +
C(onfirm)	Prompt for confirma	 of each  sets  [<disc spec.>] )}
V(erbose)	Print informa	 on each    =Landscape [<XScale> [<YScale> [<Margin> [<Threshold>]]]]] 0)used(print a hard copy ofscreen on EPSON- #.
Op	s: (use ~(force off, eg. ~ printe 	> 	 select xpression  [ sprite  displays free space  {off} 	library parameter object  all  disc  to   is   