# See the file LICENSE for redistribution information.
#
# Copyright (c) 1996, 1997, 1998, 1999
#	Sleepycat Software.  All rights reserved.
#
#	$Id: test032.tcl,v 1.1.1.6.2.2 2000/02/08 00:48:16 noriko Exp $
#
# DB Test 32 {access method}
# Use the first 10,000 entries from the dictionary.
# Insert each with self as key and "ndups" duplicates
# For the data field, prepend the letters of the alphabet
# in a random order so that we force the duplicate sorting
# code to do something.
# By setting ndups large, we can make this an off-page test
# After all are entered; test the DB_GET_BOTH functionality
# first by retrieving each dup in the file explicitly.  Then
# test the failure case.
proc test032 { method {nentries 10000} {ndups 5} {tnum 32} args } {
	global alphabet rand_init
	source ./include.tcl

	set args [convert_args $method $args]
	set omethod [convert_method $method]

	berkdb srand $rand_init

	# Create the database and open the dictionary
	set eindex [lsearch -exact $args "-env"]
	#
	# If we are using an env, then testfile should just be the db name.
	# Otherwise it is the test directory and the name.
	if { $eindex == -1 } {
		set testfile $testdir/test0$tnum.db
		set checkdb $testdir/checkdb.db
	} else {
		set testfile test0$tnum.db
		set checkdb checkdb.db
	}
	set t1 $testdir/t1
	set t2 $testdir/t2
	set t3 $testdir/t3
	cleanup $testdir

	puts "Test0$tnum:\
	    $method ($args) $nentries small sorted dup key/data pairs"
	if { [is_record_based $method] == 1 || \
	    [is_rbtree $method] == 1 } {
		puts "Test0$tnum skipping for method $omethod"
		return
	}
	set db [eval {berkdb open -create -truncate -mode 0644 \
	    $omethod -dup -dupsort} $args {$testfile} ]
	error_check_good dbopen [is_valid_db $db] TRUE
	set did [open $dict]

	set check_db [eval {berkdb \
	    open -create -truncate -mode 0644} $args {-hash $checkdb}]
	error_check_good dbopen:check_db [is_valid_db $check_db] TRUE

	set pflags ""
	set gflags ""
	set txn ""
	set count 0

	# Here is the loop where we put and get each key/data pair
	puts "\tTest0$tnum.a: Put/get loop"
	set dbc [eval {$db cursor} $txn]
	error_check_good cursor_open [is_substr $dbc $db] 1
	while { [gets $did str] != -1 && $count < $nentries } {
		# Re-initialize random string generator
		t32_random_init $ndups 

		set dups ""
		for { set i 1 } { $i <= $ndups } { incr i } {
			set pref [t32_random]
			set dups $dups$pref
			set datastr $pref:$str
                        set ret [eval {$db put} \
			    $txn $pflags {$str [chop_data $method $datastr]}]
			error_check_good put $ret 0
		}
                set ret [eval {$check_db put} \
		    $txn $pflags {$str [chop_data $method $dups]}]
		error_check_good checkdb_put $ret 0

		# Now retrieve all the keys matching this key
		set x 0
		set lastdup ""
		for {set ret [$dbc get -set $str]} \
		    {[llength $ret] != 0} \
		    {set ret [$dbc get -nextdup] } {
			set k [lindex [lindex $ret 0] 0]
			if { [string compare $k $str] != 0 } {
				break
			}
			set datastr [lindex [lindex $ret 0] 1]
			if {[string length $datastr] == 0} {
				break
			}
			if {[string compare $lastdup $datastr] > 0} {
				error_check_good sorted_dups($lastdup,$datastr)\
				    0 1
			}
			incr x
			set lastdup $datastr
		}

		error_check_good "Test0$tnum:ndups:$str" $x $ndups
		incr count
	}
	error_check_good cursor_close [$dbc close] 0
	close $did

	# Now we will get each key from the DB and compare the results
	# to the original.
	puts "\tTest0$tnum.b: Checking file for correct duplicates (no cursor)"
        set check_c [eval {$check_db cursor} $txn]
        error_check_good check_c_open(2) \
	    [is_substr $check_c $check_db] 1

	for {set ndx 0} {$ndx < [expr 4 * $ndups]} {incr ndx 4} {
		for {set ret [$check_c get -first]} \
		    {[llength $ret] != 0} \
		    {set ret [$check_c get -next] } {
			set k [lindex [lindex $ret 0] 0]
			set d [lindex [lindex $ret 0] 1]
			error_check_bad key_check:$k [string length $k] 0
			error_check_bad data_check:$d [string length $d] 0

			set pref [string range $d $ndx [expr $ndx + 3]]
			set data $pref:$k
                        set ret [eval {$db get} $txn {-get_both $k $data}]
			error_check_good \
			    get_both_data:$k $ret [list [list $k $data]]
		}
	}

	$db sync
	# Now repeat the above test using cursor ops
	puts "\tTest0$tnum.c: Checking file for correct duplicates (cursor)"
	set dbc [eval {$db cursor} $txn]
	error_check_good cursor_open [is_substr $dbc $db] 1

	for {set ndx 0} {$ndx < [expr 4 * $ndups]} {incr ndx 4} {
		for {set ret [$check_c get -first]} \
      		    {[llength $ret] != 0} \
		    {set ret [$check_c get -next] } {
			set k [lindex [lindex $ret 0] 0]
			set d [lindex [lindex $ret 0] 1]
			error_check_bad key_check:$k [string length $k] 0
			error_check_bad data_check:$d [string length $d] 0

			set pref [string range $d $ndx [expr $ndx + 3]]
			set data $pref:$k
                        set ret [eval {$dbc get} {-get_both $k $data}]
			error_check_good \
			    get_both_key:$k $ret [list [list $k $data]]
		}
	}

	# Now check the error case
	puts "\tTest0$tnum.d: Check error case (no cursor)"
	for {set ret [$check_c get -first]} \
	    {[llength $ret] != 0} \
	    {set ret [$check_c get -next] } {
		set k [lindex [lindex $ret 0] 0]
		set d [lindex [lindex $ret 0] 1]
		error_check_bad key_check:$k [string length $k] 0
		error_check_bad data_check:$d [string length $d] 0

		set data XXX$k
		set ret [eval {$db get} $txn {-get_both $k $data}]
		error_check_good error_case:$k [llength $ret] 0
	}

	# Now check the error case
	puts "\tTest0$tnum.e: Check error case (cursor)"
	for {set ret [$check_c get -first]} \
	    {[llength $ret] != 0} \
	    {set ret [$check_c get -next] } {
		set k [lindex [lindex $ret 0] 0]
		set d [lindex [lindex $ret 0] 1]
		error_check_bad key_check:$k [string length $k] 0
		error_check_bad data_check:$d [string length $d] 0

		set data XXX$k
		set ret [eval {$dbc get} {-get_both $k $data}]
		error_check_good error_case:$k [llength $ret] 0
	}

	error_check_good check_c:close [$check_c close] 0
	error_check_good check_db:close [$check_db close] 0

	error_check_good dbc_close [$dbc close] 0
	error_check_good db_close [$db close] 0
}


# We no longer support duplicate duplicates in sorted-dup dbs, so 
# we need some way to generate a whole bunch of random, yet distinct,
# data which we'll be able to store the salient features of in checkdb.
# So we write functions to generate up to $i random five-character strings,
# which should be quite ample for any number of dups we wish to generate,
# and return a random one of them, guaranteeing that we don't reuse any.
proc t32_random_init { i } {
	global t32_randlist alphabet

	# Fail if we can't generate sufficient unique strings.
	if { $i > [expr 26 * 26 * 26 * 26 * 26] } {
		set errstring\
		    "Duplicate set too large for random string generator"
		puts "FAIL:[timestamp] $errstring"
		return -code return $errstring
	}
		
	set t32_randlist {}

	# generate alphabet array
	for { set j 0 } { $j < 26 } { incr j } {
		set a($j) [string index $alphabet $j]
	}


	# Generate a list with $i elements, { aaaa, aaab, ... aaaz, aaba ...}
	for { set d1 0 ; set j 0 } { $d1 < 26 && $j < $i } { incr d1 } {
		for { set d2 0 } { $d2 < 26 && $j < $i } { incr d2 } {
			for { set d3 0 } { $d3 < 26 && $j < $i } { incr d3 } {
				for { set d4 0 } { $d4 < 26 && $j < $i } \
				    { incr d4 } {
					lappend t32_randlist \
						$a($d1)$a($d2)$a($d3)$a($d4)
					incr j
				}
			}
		}
	}

	# Randomize the list.
	for { set j 0 } { $j < $i } { incr j } {
		# Pick a random element from $j to the end	
		set k [berkdb random_int $j [expr $i - 1]]

		# Swap it with element $j
		set t1 [lindex $t32_randlist $j]
		set t2 [lindex $t32_randlist $k]

		
		set t32_randlist [lreplace $t32_randlist $j $j $t2]
		set t32_randlist [lreplace $t32_randlist $k $k $t1]
	}
}


proc t32_random {} {
	global t32_randlist 

	if { [info exists t32_randlist] == 0 || [llength $t32_randlist] == 0 } {
		set errstring "t32_random uninitialized or used too often"
		puts "FAIL:[timestamp] $errstring"
		return -code return $errstring
	}

	set item [lindex $t32_randlist 0]
	set t32_randlist [lreplace $t32_randlist 0 0]

	return $item
}
